# coding: utf-8

import os
import json
import argparse
from time import time
from base64 import urlsafe_b64encode
import subprocess

import yaml
import paramiko
import requests
from walle_api import WalleClient

TVM_ID = 2013412
TVM_URL = "https://tvm-api.yandex.net/2/ticket/"


def make_signs(data):
    res = []
    for key in paramiko.Agent().get_keys():
        res.append(urlsafe_b64encode(key.sign_ssh_data(data)))
    return res


def get_service_ticket(user=None):
    login = user or os.getlogin()
    timestamp = int(time())
    payload = "{}|{}|{}".format(timestamp, TVM_ID, TVM_ID)
    for signed_payload in make_signs(payload):
        data = {
            "grant_type": "sshkey",
            "src": TVM_ID,
            "dst": TVM_ID,
            "login": login,
            "ts": timestamp,
            "ssh_sign": signed_payload
        }
        resp = requests.post(TVM_URL, data="&".join("{}={}".format(key, value) for key, value in data.items()))
        try:
            resp.raise_for_status()
        except Exception:
            continue
        return resp.json()["2013412"]["ticket"]
    raise Exception("no ssh key found")


def get_instances(path, service_ticket, data):
    url = "https://instance-resolver-api.in.yandex-team.ru{}".format(path)
    resp = requests.post(url, headers={"X-Ya-Service-Ticket": service_ticket}, data=json.dumps(data))
    resp.raise_for_status()
    return resp.json()


def main():
    parser = argparse.ArgumentParser(description="instance resolver", add_help=True)
    parser.add_argument("--user", dest="user", help="Login for TVM")
    parser.add_argument("--field", dest="fields", help="Fields to request", nargs="*")
    parser.add_argument("-e", "--evict", dest="evict", action="store_true", help="Evict all pods from nodes")
    group = parser.add_mutually_exclusive_group(required=True)
    group.add_argument("--host", dest="host", help="Show responsibles for host")
    group.add_argument("--switch", dest="switch", help="Show responsibles for switch")
    group.add_argument("--scenario", dest="scenario", help="Show responsibles for scenario")

    args = parser.parse_args()

    instances = None
    field_mask = {"paths": args.fields or []}
    service_ticket = get_service_ticket(args.user)
    if args.host is not None:
        instances = get_instances("/v1/host_instances", service_ticket, data={
            "HostName": args.host,
            "FieldMask": field_mask
        })
        print(yaml.safe_dump(instances, default_flow_style=False))
    elif args.switch is not None:
        instances = get_instances("/v1/switch_instances", service_ticket, data={
            "SwitchName": args.switch,
            "FieldMask": field_mask
        })
        print(yaml.safe_dump(instances, default_flow_style=False))
    elif args.scenario is not None:
        walle_client = WalleClient()
        scenario = walle_client.get_scenario(args.scenario)
        invs = [host["inv"] for host in scenario["hosts"]]
        hosts = walle_client.iter_hosts(invs=invs, fields=["name"])
        for host in hosts:
            instances = get_instances("/v1/host_instances", service_ticket, data={
                "HostName": host["name"],
                "FieldMask": field_mask
            })
            print(yaml.safe_dump(instances, default_flow_style=False))
    else:
        raise Exception("no host or switch provided")

    if args.host and args.evict:
        for instance in instances["Instances"]:
            cluster = None
            if instance["Origin"] == "YP_SAS_TEST":
                cluster = "sas-test"
            elif instance["Origin"] == "YP_MAN_PRE":
                cluster = "man-pre"
            elif instance["Origin"] == "YP_SAS":
                cluster = "sas"
            elif instance["Origin"] == "YP_MAN":
                cluster = "man"
            elif instance["Origin"] == "YP_VLA":
                cluster = "vla"

            if cluster is None:
                print("Can't deduce yp cluster for {!r}".format(instance))
                continue

            pod_id = instance["ContainerFqdn"].split(".", 1)[0]
            print("Evicting pod {}".format(pod_id))

            try:
                subprocess.check_call([
                    "ya", "tool", "yp", "acknowledge-eviction",
                    pod_id, 'instance resolver cli', "--address", cluster,
                    "--address", cluster
                ])
            except subprocess.CalledProcessError as exc:
                print(exc)

    elif args.evict:
        raise Exception("evict should be used only with host")


if __name__ == "__main__":
    main()
