package main

import (
	"crypto/tls"
	"flag"
	"net"
	"net/http"
	"os"
	"os/signal"
	"strings"
	"syscall"
	"time"

	"github.com/grpc-ecosystem/grpc-gateway/runtime"
	"go.uber.org/zap/zapcore"
	"golang.org/x/net/context"
	"google.golang.org/grpc"
	"google.golang.org/grpc/metadata"

	"a.yandex-team.ru/infra/rtc/instance_resolver/internal/server"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/abc"
	auth2 "a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/auth"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/bot"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/golovan"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/iss3"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/nanny"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/netmon"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/qloud"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/qyp"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/staff"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/walle"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/yp"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/freshcache"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/log"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/util"

	aLog "a.yandex-team.ru/library/go/core/log"
)

var (
	grpcPort        = flag.String("grpc-endpoint", "localhost:8787", "endpoint of GRPC API")
	restPort        = flag.String("rest-endpoint", ":8080", "endpoint of http REST API")
	restHTTPSPort   = flag.String("rest-https-endpoint", "", "endpoint of https REST API")
	ypPort          = flag.String("yp-endpoint", "localhost:9367", "endpoint of YP Proxy")
	specPath        = flag.String("spec-path", "swagger.json", "path to swagger spec")
	nannyToken      = flag.String("nanny-token", os.Getenv("NANNY_TOKEN"), "OAuth token for Nanny")
	qypToken        = flag.String("qyp-token", os.Getenv("QYP_TOKEN"), "OAuth token for QYP")
	qloudToken      = flag.String("qloud-token", os.Getenv("QLOUD_TOKEN"), "OAuth token for Qloud")
	staffToken      = flag.String("staff-token", os.Getenv("STAFF_TOKEN"), "OAuth token for Staff")
	walleToken      = flag.String("walle-token", os.Getenv("WALLE_TOKEN"), "OAuth token for Wall-E")
	tvmToken        = flag.String("tvm-token", os.Getenv("QLOUD_TVM_TOKEN"), "TVM token")
	tvmPort         = flag.String("tvm-port", os.Getenv("QLOUD_TVM_PORT"), "TVM port")
	certificatePath = flag.String("certificate-path", "", "path to iss certificate")
	httpsCertPath   = flag.String("https-cert-path", "server.pem", "path to server certificate")
	httpsKeyPath    = flag.String("https-key-path", "server.key", "path to server key")
	cacheDir        = flag.String("cache-dir", ".", "path to cache directory")
)

func main() {
	flag.Parse()

	sigs := make(chan os.Signal, 1)
	signal.Notify(sigs, syscall.SIGINT, syscall.SIGTERM)

	ctx := context.Background()
	ctx, cancel := context.WithCancel(ctx)
	defer cancel()

	var cert *tls.Certificate
	var err error
	if *certificatePath != "" {
		cert, err = util.LoadCertficateAndKey(*certificatePath)
		if err != nil {
			log.Logger.Error("failed to load certificate", aLog.Error(err))
			return
		}
	}

	authClient := auth2.NewAuthClient(*tvmToken, *tvmPort)
	issFactory := iss3.NewIssCacherFactory()

	hostname, err := os.Hostname()
	if err != nil {
		log.Logger.Error("failed to get host name", aLog.Error(err))
		return
	}

	netmonClient, err := netmon.NewNetmonClient()
	if err != nil {
		log.Logger.Error("failed to create client to netmon", aLog.Error(err))
		return
	}

	nannyClient, err := nanny.NewNannyClient(*nannyToken)
	if err != nil {
		log.Logger.Error("failed to create client to nanny", aLog.Error(err))
		return
	}

	qypClient, err := qyp.NewQypClient(*qypToken)
	if err != nil {
		log.Logger.Error("failed to create client to QYP", aLog.Error(err))
		return
	}

	qloud.ConfigureClients(*qloudToken)

	staffClient, err := staff.NewStaffClient(*staffToken)
	if err != nil {
		log.Logger.Error("failed to create client to staff", aLog.Error(err))
		return
	}

	abcClient, err := abc.NewAbcClient(*staffToken)
	if err != nil {
		log.Logger.Error("failed to create client to abc", aLog.Error(err))
		return
	}

	staffCacheDriver := staff.NewCacheDriver(staffClient, abcClient)

	ypClient, err := yp.NewYpClient(*ypPort)
	if err != nil {
		log.Logger.Error("failed to create client to yp", aLog.Error(err))
		return
	}
	defer ypClient.Close()

	botClient := bot.NewBotClient()
	botCacheDriver := bot.NewCacheDriver(botClient)

	golovanClient := golovan.NewGolovanClient()
	golovanCacheDriver := golovan.NewCacheDriver(golovanClient)

	walleClient := walle.NewWalleClient(*walleToken)
	walleCacheDriver := walle.NewCacheDriver(walleClient)

	freshCache := freshcache.NewCache(*cacheDir)
	freshCache.RegisterDriver(botCacheDriver)
	freshCache.RegisterDriver(golovanCacheDriver)
	freshCache.RegisterDriver(staffCacheDriver)
	freshCache.RegisterDriver(walleCacheDriver)
	freshCache.Start()
	defer freshCache.Stop()

	lis, err := net.Listen("tcp6", *grpcPort)
	if err != nil {
		log.Logger.Error("failed to listen", aLog.Error(err))
		return
	}

	authChecker := func(ctx context.Context, r *http.Request) metadata.MD {
		attributes := make(map[string]string)
		if login, ok := authClient.CheckAuth(ctx, r); ok {
			attributes["authorized"] = "true"
			attributes["login"] = login
		} else {
			attributes["authorized"] = "false"
		}
		if r.RequestURI == "/v1/ping" {
			attributes["ping"] = "true"
		}
		return metadata.New(attributes)
	}

	runtime.DefaultContextTimeout = 5 * time.Minute
	mux := runtime.NewServeMux(runtime.WithMetadata(authChecker))
	opts := []grpc.DialOption{grpc.WithInsecure()}
	if err := server.RegisterTResolverHandlerFromEndpoint(ctx, mux, *grpcPort, opts); err != nil {
		log.Logger.Error("failed to RegisterTResolverHandlerFromEndpoint", aLog.Error(err))
		return
	}
	server.NewSwagUI(*specPath, mux)
	server.NewUnistatHandle("stats", mux)

	gRPCServer := grpc.NewServer(
		grpc.UnaryInterceptor(grpc.UnaryServerInterceptor(server.RequestInterceptor)),
	)
	resolverServer := server.ResolverServer{
		Certificate:   cert,
		NetmonClient:  netmonClient,
		IssFactory:    issFactory,
		YpClient:      ypClient,
		NannyClient:   nannyClient,
		QypClient:     qypClient,
		StaffClient:   staffClient,
		BotClient:     botClient,
		WalleClient:   walleClient,
		GolovanClient: golovanClient,
	}
	resolverServer.Register(gRPCServer)

	log.Level.SetLevel(zapcore.InfoLevel)

	rewriteHeaders := func(h http.Handler) http.Handler {
		return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if origin := r.Header.Get("Origin"); origin != "" {
				w.Header().Set("Access-Control-Allow-Origin", origin)
				w.Header().Set("Access-Control-Allow-Credentials", "true")
				if r.Method == "OPTIONS" && r.Header.Get("Access-Control-Request-Method") != "" {
					headers := []string{"Content-Type", "Accept"}
					w.Header().Set("Access-Control-Allow-Headers", strings.Join(headers, ","))
					methods := []string{"GET", "HEAD", "POST", "PUT", "DELETE"}
					w.Header().Set("Access-Control-Allow-Methods", strings.Join(methods, ","))
					return
				}
			}
			w.Header().Set("X-Backend-Server", hostname)
			h.ServeHTTP(w, r)
		})
	}

	go func() {
		if err := http.ListenAndServe(*restPort, rewriteHeaders(mux)); err != nil {
			log.Logger.Error("failed to serve http proxy", aLog.Error(err))
			gRPCServer.Stop()
		}
	}()

	if *restHTTPSPort != "" {
		go func() {
			if err := http.ListenAndServeTLS(*restHTTPSPort, *httpsCertPath, *httpsKeyPath, rewriteHeaders(mux)); err != nil {
				log.Logger.Error("failed to serve https proxy", aLog.Error(err))
				gRPCServer.Stop()
			}
		}()
	}

	go func() {
		<-sigs
		log.Logger.Info("stopping by signal")
		gRPCServer.Stop()
	}()

	if err := gRPCServer.Serve(lis); err != nil {
		log.Logger.Error("failed to serve", aLog.Error(err))
		return
	}
}
