package server

import (
	"fmt"
	"strings"
	"sync"
	"time"

	"golang.org/x/net/context"
	"google.golang.org/genproto/protobuf/field_mask"

	pb "a.yandex-team.ru/infra/rtc/instance_resolver/api"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/deploy"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/iss3"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/nanny"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/netmon"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/qloud"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/qyp"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/staff"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/yp"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/log"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/util"
	aLog "a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/unistat"
)

const backendThreads = 4
const nannyThreads = 6
const qypThreads = 3
const qloudThreads = 3
const deployThreads = 3

type OriginFailure struct {
	Origin string
	Error  error
}

type InstanceCollector struct {
	ctx            context.Context
	resolverServer *ResolverServer

	hosts       []*pb.THost
	failedHosts map[string]string

	originGroup      sync.WaitGroup
	platformGroup    sync.WaitGroup
	postprocessGroup sync.WaitGroup

	failureChannel   chan OriginFailure
	instancesChannel chan *pb.TInstance
	nodesChannel     chan *yp.YpNode
	groupsChannel    chan string
	loginsChannel    chan string

	failedOrigins map[string]string
	instances     []*pb.TInstance
	ypNodes       map[string]*yp.YpNode

	nannySemaphore *util.SimpleSemaphore
	nannyState     *nanny.NannyState

	qypSemaphore *util.SimpleSemaphore
	qypState     *qyp.QypState

	qloudSemaphore *util.SimpleSemaphore
	qloudState     *qloud.QloudState

	deploySemaphore *util.SimpleSemaphore
	deployState     *deploy.DeployState

	staffState *staff.StaffState

	requestedFields []string
}

func NewInstanceCollector(ctx context.Context, resolverServer *ResolverServer, hostsInfo []netmon.NetmonHostInfo, fieldMask *field_mask.FieldMask) (collector *InstanceCollector) {
	collector = &InstanceCollector{
		ctx:              ctx,
		resolverServer:   resolverServer,
		failureChannel:   make(chan OriginFailure, 10),
		instancesChannel: make(chan *pb.TInstance, 100),
		nodesChannel:     make(chan *yp.YpNode, 100),
		groupsChannel:    make(chan string, 100),
		loginsChannel:    make(chan string, 100),
		failedOrigins:    make(map[string]string),
		instances:        make([]*pb.TInstance, 0),
		ypNodes:          make(map[string]*yp.YpNode),
		nannySemaphore:   util.NewSimpleSemaphore(nannyThreads),
		nannyState:       nanny.NewNannyState(),
		qypSemaphore:     util.NewSimpleSemaphore(qypThreads),
		qypState:         qyp.NewQypState(),
		qloudSemaphore:   util.NewSimpleSemaphore(qloudThreads),
		qloudState:       qloud.NewQloudState(),
		deploySemaphore:  util.NewSimpleSemaphore(deployThreads),
		deployState:      deploy.NewDeployState(),
		staffState:       staff.NewStaffState(),
		requestedFields:  make([]string, 0),
	}
	hosts, failedHosts := collector.createHosts(hostsInfo)
	collector.hosts = hosts
	collector.failedHosts = failedHosts
	if fieldMask != nil {
		collector.requestedFields = fieldMask.Paths
	}
	return
}

func (collector *InstanceCollector) filterYasmTags(tags map[string]string) map[string]string {
	if itype, ok := tags["itype"]; ok {
		itypeInfo, err := collector.resolverServer.GolovanClient.GetItypeInfo(itype)
		if itypeInfo == nil || err != nil {
			return tags
		}
		result := make(map[string]string)
		result["itype"] = itype
		for _, tag := range itypeInfo.Tags {
			if val, ok := tags[tag]; ok {
				result[tag] = val
			}
		}
		return result
	}
	return tags
}

func (collector *InstanceCollector) examineIssCacher(cacherAddr iss3.IssCacherAddr, hosts []string) {
	defer collector.originGroup.Done()

	issClient, err := collector.resolverServer.IssFactory.NewClient(collector.resolverServer.Certificate, cacherAddr.Host, cacherAddr.Port)
	if err != nil {
		defer collector.resolverServer.UnistatIssErrors.Update(1)
		collector.failureChannel <- OriginFailure{cacherAddr.Origin, err}
		log.Logger.Error("failed to connect to iss", aLog.Reflect("iss", cacherAddr), aLog.Error(err))
		return
	}
	defer func() { _ = issClient.Close() }()

	for _, host := range hosts {
		defer unistat.MeasureMicrosecondsSince(collector.resolverServer.UnistatIssLatency, time.Now())
		defer collector.resolverServer.UnistatIssCalls.Update(1)

		instances, err := issClient.GetHostInstances(collector.ctx, host)
		if err != nil {
			defer collector.resolverServer.UnistatIssErrors.Update(1)
			collector.failureChannel <- OriginFailure{cacherAddr.Origin, err}
			log.Logger.Error("failed to request host from iss", aLog.String("host", host), aLog.Reflect("iss", cacherAddr), aLog.Error(err))
			continue
		}

		for _, instance := range instances {
			collector.instancesChannel <- &pb.TInstance{
				IssSlot:            instance.Slot,
				IssConfiguration:   instance.Configuration,
				NannyServiceId:     instance.NannyServiceId,
				FromNanny:          instance.FromNanny,
				FromQloud:          instance.FromQloud,
				QloudComponent:     instance.QloudComponent,
				QloudInstallation:  instance.QloudInstallation,
				GencfgGroup:        instance.GencfgGroup,
				Origin:             cacherAddr.Origin,
				Host:               host,
				ContainerFqdn:      instance.ContainerFqdn,
				YasmTags:           collector.filterYasmTags(instance.YasmTags),
				HasSlotCpuLimit:    instance.HasSlotCpuLimit,
				HasSlotMemoryLimit: instance.HasSlotMemoryLimit,
			}
		}
	}
}

func (collector *InstanceCollector) examineYpBackend(ypAddr yp.YpAddr, hosts []string) {
	defer collector.originGroup.Done()

	for _, host := range hosts {
		defer unistat.MeasureMicrosecondsSince(collector.resolverServer.UnistatYpLatency, time.Now())
		defer collector.resolverServer.UnistatYpCalls.Update(1)

		instances, err := collector.resolverServer.YpClient.GetHostInstances(collector.ctx, host, ypAddr.Origin)
		if err != nil {
			defer collector.resolverServer.UnistatYpErrors.Update(1)
			collector.failureChannel <- OriginFailure{ypAddr.Origin, err}
			log.Logger.Error("failed to request pods from yp", aLog.String("host", host), aLog.Reflect("yp", ypAddr), aLog.Error(err))
			continue
		}

		for _, instance := range instances {
			collector.instancesChannel <- &pb.TInstance{
				YpPodSet:           instance.YpPodSet,
				NannyServiceId:     instance.NannyServiceID,
				FromQyp:            instance.FromQyp,
				FromNanny:          instance.FromNanny,
				FromDeploy:         instance.FromDeploy,
				Origin:             ypAddr.Origin,
				Host:               host,
				ContainerFqdn:      instance.ContainerFqdn,
				DeployEngine:       instance.DeployEngine,
				MaintenanceState:   instance.MaintenanceState,
				YasmTags:           collector.filterYasmTags(instance.YasmTags),
				HasSlotCpuLimit:    true,
				HasSlotMemoryLimit: true,
				DeployEnviron:      instance.DeployEnviron,
				DeployUnitId:       instance.DeployUnitID,
				DeployStageId:      instance.DeployStageID,
				DeployProjectId:    instance.DeployProjectID,
			}
		}

		node, err := collector.resolverServer.YpClient.GetNode(collector.ctx, host, ypAddr.Origin)
		if err != nil {
			defer collector.resolverServer.UnistatYpErrors.Update(1)
			collector.failureChannel <- OriginFailure{ypAddr.Origin, err}
			log.Logger.Error("failed to request node from yp", aLog.String("host", host), aLog.Reflect("yp", ypAddr), aLog.Error(err))
			continue
		}
		collector.nodesChannel <- &node
	}
}

func (collector *InstanceCollector) examineNannyService(nannyServiceID string) {
	defer collector.platformGroup.Done()

	collector.nannySemaphore.Acquire()
	defer collector.nannySemaphore.Release()

	defer unistat.MeasureMicrosecondsSince(collector.resolverServer.UnistatNannyLatency, time.Now())
	defer collector.resolverServer.UnistatNannyCalls.Update(1)

	infoAttrs, err := collector.resolverServer.NannyClient.GetServiceInfoAttrs(collector.ctx, nannyServiceID)
	if err != nil {
		defer collector.resolverServer.UnistatNannyErrors.Update(1)
		collector.nannyState.SetError(nannyServiceID, err)
		return
	}
	if infoAttrs == nil {
		return
	}

	authAttrs, err := collector.resolverServer.NannyClient.GetServiceAuthAttrs(collector.ctx, nannyServiceID)
	if err != nil {
		defer collector.resolverServer.UnistatNannyErrors.Update(1)
		collector.nannyState.SetError(nannyServiceID, err)
		return
	}
	if authAttrs == nil {
		return
	}

	pbService := &pb.TNannyService{
		ServiceId:    nannyServiceID,
		Owners:       util.CreateNannyPersons(&authAttrs.Owners),
		ConfManagers: util.CreateNannyPersons(&authAttrs.ConfManagers),
		OpsManagers:  util.CreateNannyPersons(&authAttrs.OpsManagers),
		Observers:    util.CreateNannyPersons(&authAttrs.Observers),
		ServiceUrl:   fmt.Sprintf("https://nanny.yandex-team.ru/ui/#/services/catalog/%s/", nannyServiceID),
	}
	if infoAttrs.AbcGroup != nil {
		pbService.AbcGroup = *infoAttrs.AbcGroup
	}
	if infoAttrs.MaintenanceNotifications != nil {
		pbService.MaintenanceNotifications = *infoAttrs.MaintenanceNotifications
	}
	collector.nannyState.Append(pbService)

	// resolve found logins and groups
	for _, groupID := range util.GetNannyGroups(pbService) {
		if groupID != "" {
			collector.groupsChannel <- groupID
		}
	}
	for _, login := range util.GetNannyLogins(pbService) {
		if login != "" {
			collector.loginsChannel <- login
		}
	}
}

func (collector *InstanceCollector) examineQloudDNSSD(dnsSD string) {
	defer collector.platformGroup.Done()

	collector.qloudSemaphore.Acquire()
	defer collector.qloudSemaphore.Release()

	defer unistat.MeasureMicrosecondsSince(collector.resolverServer.UnistatQloudLatency, time.Now())
	defer collector.resolverServer.UnistatQloudCalls.Update(1)

	qloudEnv, err := qloud.NewEnvironmentFromDNSSD(dnsSD)
	if err != nil {
		collector.qloudState.SetError(dnsSD, err)
		collector.resolverServer.UnistatQloudErrors.Update(1)
		return
	}

	acl, err := qloud.GetClient(qloudEnv).CollectACL(collector.ctx, qloudEnv)

	if err != nil {
		collector.qloudState.SetError(dnsSD, err)
		collector.resolverServer.UnistatQloudErrors.Update(1)
		return
	}

	pbEnv := &pb.TQloudEnvironment{
		DnsSD:      dnsSD,
		ACL:        collectQloudAccessors(acl),
		ServiceUrl: qloudEnv.GetURL(),
	}
	collector.qloudState.Append(pbEnv)

	for _, qloudACE := range pbEnv.ACL.ACEs {
		if strings.EqualFold(qloudACE.Type, "user") {
			collector.loginsChannel <- qloudACE.Id
		} else {
			collector.groupsChannel <- qloudACE.Id
		}
	}
}

func (collector *InstanceCollector) examineQypVM(origin string, podset string) {
	defer collector.platformGroup.Done()

	collector.qypSemaphore.Acquire()
	defer collector.qypSemaphore.Release()

	defer unistat.MeasureMicrosecondsSince(collector.resolverServer.UnistatQypLatency, time.Now())
	defer collector.resolverServer.UnistatQypCalls.Update(1)

	owners, err := collector.resolverServer.QypClient.GetOwners(collector.ctx, origin, podset)
	if err != nil {
		defer collector.resolverServer.UnistatQypErrors.Update(1)
		collector.qypState.SetError(podset, err)
		return
	}
	if owners == nil {
		return
	}

	pbVM := &pb.TQypVM{
		VmId:    podset,
		Cluster: collector.resolverServer.QypClient.GetClusterName(origin),
		Owners:  util.CreateQypOwners(owners),
	}
	pbVM.ServiceUrl = fmt.Sprintf("https://qyp.yandex-team.ru/vm/%s/%s", pbVM.Cluster, pbVM.VmId)
	collector.qypState.Append(pbVM)

	// resolve found logins and groups
	for _, groupID := range util.GetQypGroups(pbVM) {
		collector.groupsChannel <- groupID
	}
	for _, login := range util.GetQypLogins(pbVM) {
		collector.loginsChannel <- login
	}
}

func (collector *InstanceCollector) examineDeployDU(origin string, projectID string, stageID string, deployUnitID string, deployEnviron string) {
	defer collector.platformGroup.Done()

	collector.deploySemaphore.Acquire()
	defer collector.deploySemaphore.Release()

	defer unistat.MeasureMicrosecondsSince(collector.resolverServer.UnistatDeployLatency, time.Now())
	defer collector.resolverServer.UnistatDeployCalls.Update(1)

	domain := deploy.GetDomainName(deployEnviron)
	cluster := deploy.GetClusterName(origin)

	pbDeployUnit := &pb.TDeployDU{
		Cluster:      cluster,
		DeployUnitId: deployUnitID,
		StageId:      stageID,
		ProjectId:    projectID,
		ServiceUrl:   fmt.Sprintf("https://%s/stage/%s/status/%s/%s", domain, stageID, deployUnitID, cluster),
	}
	collector.deployState.Append(pbDeployUnit)
}

func collectQloudAccessors(acl *qloud.ACL) *pb.TQloudACL {
	result := &pb.TQloudACL{ObjectId: acl.ObjectID,
		ObjectName: acl.ObjectName,
		ACEs:       make([]*pb.TQloudACE, 0),
	}

	for _, ace := range acl.ACEs {
		// append explicit users to user and all other to groups
		qloudACE := pb.TQloudACE{
			GrantedObjectId:    ace.GrantedObjectID,
			User:               ace.User,
			Permission:         ace.Permission,
			Admin:              fmt.Sprintf("%t", ace.Admin),
			Type:               ace.Type,
			Id:                 fmt.Sprintf("%d", ace.ID),
			RoleId:             fmt.Sprintf("%d", ace.RoleID),
			GrantedObjectLevel: ace.GrantedObjectLevel,
			GrantedObjectName:  ace.GrantedObjectName,
		}
		result.ACEs = append(result.ACEs, &qloudACE)
	}
	return result
}

func (collector *InstanceCollector) examineStaffGroup(groupID string) {
	defer collector.postprocessGroup.Done()

	defer unistat.MeasureMicrosecondsSince(collector.resolverServer.UnistatStaffLatency, time.Now())
	defer collector.resolverServer.UnistatStaffCalls.Update(1)

	persons, err := collector.resolverServer.StaffClient.GetPersonsInGroup(groupID)
	if err != nil {
		defer collector.resolverServer.UnistatStaffErrors.Update(1)
		collector.staffState.SetGroupError(groupID, err)
		return
	}

	collector.staffState.SetPersonsInGroup(groupID, persons)
}

func (collector *InstanceCollector) examineLogin(login string) {
	defer collector.postprocessGroup.Done()

	defer unistat.MeasureMicrosecondsSince(collector.resolverServer.UnistatStaffLatency, time.Now())
	defer collector.resolverServer.UnistatStaffCalls.Update(1)

	person, err := collector.resolverServer.StaffClient.GetPerson(login)
	if err != nil {
		defer collector.resolverServer.UnistatStaffErrors.Update(1)
		collector.staffState.SetLoginError(login, err)
		return
	}
	if person == nil {
		return
	}

	collector.staffState.SetPerson(login, person)
}

func (collector *InstanceCollector) Spawn() {
	// spawn requests
	for origin, hosts := range util.GroupHostByIssCacher(collector.hosts) {
		if cacherAddr, ok := iss3.GetIssCacherAddr(origin); ok {
			for _, chunk := range util.SplitHosts(hosts, backendThreads) {
				collector.originGroup.Add(1)
				go collector.examineIssCacher(*cacherAddr, chunk)
			}
		}
	}

	for origin, hosts := range util.GroupHostByYpMaster(collector.hosts) {
		for _, chunk := range util.SplitHosts(hosts, backendThreads) {
			collector.originGroup.Add(1)
			go collector.examineYpBackend(yp.YpAddr{Origin: origin}, chunk)
		}
	}

	// close channels related to origin
	collector.platformGroup.Add(1)
	go func() {
		defer collector.platformGroup.Done()
		collector.originGroup.Wait()
		close(collector.instancesChannel)
		close(collector.nodesChannel)
		close(collector.failureChannel)
	}()

	// collect failures
	collector.platformGroup.Add(1)
	go func() {
		defer collector.platformGroup.Done()
		for originFailure := range collector.failureChannel {
			collector.failedOrigins[originFailure.Origin] = originFailure.Error.Error()
		}
	}()

	// collect nodes
	collector.platformGroup.Add(1)
	go func() {
		defer collector.platformGroup.Done()
		for node := range collector.nodesChannel {
			collector.ypNodes[node.NodeID] = node
		}
	}()

	// collect instances
	collector.platformGroup.Add(1)
	go func() {
		defer collector.platformGroup.Done()
		for instance := range collector.instancesChannel {
			if instance.FromNanny &&
				util.HasNannyServicesField(collector.requestedFields) &&
				!collector.nannyState.IsSeen(instance.NannyServiceId) {
				collector.platformGroup.Add(1)
				go collector.examineNannyService(instance.NannyServiceId)
			} else if instance.FromQloud &&
				util.HasQloudEnvironmentsField(collector.requestedFields) &&
				!collector.qloudState.IsSeen(instance.QloudComponent) {
				collector.platformGroup.Add(1)
				go collector.examineQloudDNSSD(instance.QloudComponent)
			} else if instance.FromQyp &&
				util.HasQypVMsField(collector.requestedFields) &&
				!collector.qypState.IsSeen(instance.YpPodSet) {
				collector.platformGroup.Add(1)
				go collector.examineQypVM(instance.Origin, instance.YpPodSet)
			} else if instance.FromDeploy &&
				util.HasDeployDUsField(collector.requestedFields) &&
				!collector.deployState.IsSeen(instance.DeployStageId, instance.DeployUnitId) {
				collector.platformGroup.Add(1)
				go collector.examineDeployDU(instance.Origin, instance.DeployProjectId, instance.DeployStageId, instance.DeployUnitId, instance.DeployEnviron)
			}
			collector.instances = append(collector.instances, instance)
		}
	}()

	// close channels related to platform
	collector.postprocessGroup.Add(1)
	go func() {
		defer collector.postprocessGroup.Done()
		collector.platformGroup.Wait()
		close(collector.groupsChannel)
		close(collector.loginsChannel)
	}()

	// some post processing
	collector.postprocessGroup.Add(1)
	go func() {
		defer collector.postprocessGroup.Done()
		for groupID := range collector.groupsChannel {
			if !collector.staffState.IsGroupSeen(groupID) {
				collector.postprocessGroup.Add(1)
				go collector.examineStaffGroup(groupID)
			}
		}
	}()

	collector.postprocessGroup.Add(1)
	go func() {
		defer collector.postprocessGroup.Done()
		for login := range collector.loginsChannel {
			if !collector.staffState.IsLoginSeen(login) {
				collector.postprocessGroup.Add(1)
				go collector.examineLogin(login)
			}
		}
	}()
}

func appendUniqPerson(uniqueLogins map[string]bool, person *staff.StaffPerson) {
	if person != nil && !person.IsDeleted && !person.Official.IsDismissed && !person.Official.IsRobot && !person.Official.IsHomeworker {
		uniqueLogins[person.Login] = true
	}
}

func (collector *InstanceCollector) collectLogins(logins []string, groupIds []string) []string {
	uniqueLogins := make(map[string]bool)
	for _, groupID := range groupIds {
		if groupID != "" {
			for _, person := range collector.staffState.GetPersonsInGroup(groupID) {
				appendUniqPerson(uniqueLogins, &person)
			}
		}
	}
	for _, login := range logins {
		if login != "" {
			appendUniqPerson(uniqueLogins, collector.staffState.GetPerson(login))
		}
	}
	result := make([]string, 0, len(uniqueLogins))
	for login := range uniqueLogins {
		result = append(result, login)
	}
	return result
}

func (collector *InstanceCollector) createResponsibleGroups() (result []*pb.TResponsibleGroup) {
	result = make([]*pb.TResponsibleGroup, 0)

	for _, nannyService := range collector.nannyState.GetServices() {
		if nannyService.MaintenanceNotifications == "Disable" {
			continue
		}

		logins := make([]string, 0)
		if nannyService.AbcGroup != 0 {
			persons, err := collector.resolverServer.StaffClient.GetPersonsInService(nannyService.AbcGroup)
			if err == nil {
				for _, person := range persons {
					logins = append(logins, person.Login)
				}
			}
		}

		if len(logins) == 0 {
			logins = collector.collectLogins(nannyService.Owners.Logins, nannyService.Owners.Groups)
		}

		result = append(result, &pb.TResponsibleGroup{
			NannyServiceId:      nannyService.ServiceId,
			ServiceUrl:          nannyService.ServiceUrl,
			Logins:              logins,
			EnableNotifications: nannyService.MaintenanceNotifications == "Enable",
		})
	}

	for _, qloudEnv := range collector.qloudState.GetEnvironments() {
		envUsers, envGroups := util.GetQloudUsersAndGroups(qloudEnv)
		result = append(result, &pb.TResponsibleGroup{
			QloudComponent:      qloudEnv.DnsSD,
			ServiceUrl:          qloudEnv.ServiceUrl,
			Logins:              collector.collectLogins(envUsers, envGroups),
			EnableNotifications: false,
		})
	}

	for _, vm := range collector.qypState.GetVMs() {
		result = append(result, &pb.TResponsibleGroup{
			QypVmId:             vm.VmId,
			ServiceUrl:          vm.ServiceUrl,
			Logins:              collector.collectLogins(util.GetQypLogins(vm), util.GetQypGroups(vm)),
			EnableNotifications: true,
		})
	}

	return
}

func (collector *InstanceCollector) createHosts(hostsInfo []netmon.NetmonHostInfo) (hosts []*pb.THost, failedHosts map[string]string) {
	hosts = make([]*pb.THost, 0, len(hostsInfo))
	failedHosts = make(map[string]string)
	for _, hostInfo := range hostsInfo {
		botHostInfo, err := collector.resolverServer.BotClient.GetHostInfo(hostInfo.Name)
		if err != nil {
			failedHosts[hostInfo.Name] = err.Error()
			continue
		}
		if botHostInfo == nil {
			failedHosts[hostInfo.Name] = "not found in bot"
			continue
		}

		pbHost := &pb.THost{
			HostName:        hostInfo.Name,
			DatacenterName:  hostInfo.DatacenterName,
			QueueName:       hostInfo.QueueName,
			SwitchName:      hostInfo.SwitchName,
			InventoryNumber: botHostInfo.InventoryNumber,
			AbcServiceId:    botHostInfo.PlannerID,
		}

		walleHostInfo, err := collector.resolverServer.WalleClient.GetHostInfo(hostInfo.Name)
		if err != nil {
			failedHosts[hostInfo.Name] = err.Error()
			continue
		}
		if walleHostInfo != nil {
			pbHost.YpMaster = walleHostInfo.YpMaster
			pbHost.IssCachers = walleHostInfo.IssCachers
			pbHost.FromRtc = walleHostInfo.FromRtc
		}

		hosts = append(hosts, pbHost)
	}
	return
}

func (collector *InstanceCollector) GetInstancesReply() (reply *pb.TInstancesReply, err error) {
	collector.originGroup.Wait()
	collector.platformGroup.Wait()
	collector.postprocessGroup.Wait()

	reply = &pb.TInstancesReply{
		Instances:               collector.instances,
		FailedOrigins:           collector.failedOrigins,
		NannyServices:           collector.nannyState.GetServices(),
		FailedNannyServices:     collector.nannyState.GetFailedServices(),
		QypVMs:                  collector.qypState.GetVMs(),
		FailedQypVMs:            collector.qypState.GetFailedVMs(),
		ResponsibleGroups:       collector.createResponsibleGroups(),
		FailedStaffGroups:       collector.staffState.GetFailedGroups(),
		FailedStaffLogins:       collector.staffState.GetFailedLogins(),
		QloudEnvironments:       collector.qloudState.GetEnvironments(),
		FailedQloudEnvironments: collector.qloudState.GetFailedEnvironments(),
		DeployDUs:               collector.deployState.GetDeployUnits(),
		FailedDeployDUs:         collector.deployState.GetFailedDeployUnits(),
		Hosts:                   collector.hosts,
		FailedHosts:             collector.failedHosts,
	}

	for _, host := range reply.Hosts {
		if ypNode, ok := collector.ypNodes[host.HostName]; ok {
			host.YpSegment = ypNode.Segment
			host.YpMaintenanceState = ypNode.MaintenanceState
		}
	}

	return
}
