package server

import (
	"fmt"
	"reflect"
	"time"

	"golang.org/x/net/context"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/metadata"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/log"
	aLog "a.yandex-team.ru/library/go/core/log"
)

func authorize(md metadata.MD) error {
	if reflect.DeepEqual(md.Get("ping"), []string{"true"}) {
		return nil
	}
	if !reflect.DeepEqual(md.Get("authorized"), []string{"true"}) {
		return status.Errorf(codes.Unauthenticated, "Authorization is failed")
	}
	return nil
}

func firstOrDefault(values []string) string {
	if len(values) == 0 {
		return ""
	} else {
		return values[0]
	}
}

func RequestInterceptor(ctx context.Context, req interface{}, info *grpc.UnaryServerInfo, handler grpc.UnaryHandler) (resp interface{}, err error) {
	md, ok := metadata.FromIncomingContext(ctx)
	if !ok {
		return nil, status.Errorf(codes.InvalidArgument, "Retrieving metadata is failed")
	}
	if err := authorize(md); err != nil {
		return nil, err
	}

	startTS := time.Now()
	resp, err = handler(ctx, req)
	dur := aLog.Duration("latency", time.Since(startTS))
	authorized := aLog.String("authorized", firstOrDefault(md.Get("authorized")))
	logins := aLog.String("login", firstOrDefault(md.Get("login")))
	log.Logger.Info(fmt.Sprintf("[%v] Call %v", startTS.Format(time.RFC3339), info.FullMethod), dur, authorized, logins)
	return resp, err
}
