package server

import (
	"crypto/tls"
	"fmt"
	"time"

	"golang.org/x/net/context"
	"google.golang.org/genproto/protobuf/field_mask"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	pb "a.yandex-team.ru/infra/rtc/instance_resolver/api"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/bot"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/golovan"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/iss3"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/nanny"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/netmon"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/qyp"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/staff"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/walle"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/yp"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/log"
	aLog "a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/unistat"
	"a.yandex-team.ru/library/go/yandex/unistat/aggr"
)

type ResolverServer struct {
	Certificate   *tls.Certificate
	NetmonClient  *netmon.NetmonClient
	IssFactory    *iss3.IssFactory
	YpClient      *yp.YpClient
	NannyClient   *nanny.NannyClient
	QypClient     *qyp.QypClient
	StaffClient   *staff.StaffClient
	BotClient     *bot.BotClient
	WalleClient   *walle.WalleClient
	GolovanClient *golovan.GolovanClient

	UnistatRequests       *unistat.Numeric
	UnistatRequestLatency *unistat.Histogram

	UnistatIssCalls   *unistat.Numeric
	UnistatIssLatency *unistat.Histogram
	UnistatIssErrors  *unistat.Numeric

	UnistatYpCalls   *unistat.Numeric
	UnistatYpLatency *unistat.Histogram
	UnistatYpErrors  *unistat.Numeric

	UnistatNannyCalls   *unistat.Numeric
	UnistatNannyLatency *unistat.Histogram
	UnistatNannyErrors  *unistat.Numeric

	UnistatQypCalls   *unistat.Numeric
	UnistatQypLatency *unistat.Histogram
	UnistatQypErrors  *unistat.Numeric

	UnistatQloudCalls   *unistat.Numeric
	UnistatQloudLatency *unistat.Histogram
	UnistatQloudErrors  *unistat.Numeric

	UnistatDeployCalls   *unistat.Numeric
	UnistatDeployLatency *unistat.Histogram
	UnistatDeployErrors  *unistat.Numeric

	UnistatStaffCalls   *unistat.Numeric
	UnistatStaffLatency *unistat.Histogram
	UnistatStaffErrors  *unistat.Numeric

	UnistatWalleCalls   *unistat.Numeric
	UnistatWalleLatency *unistat.Histogram
	UnistatWalleErrors  *unistat.Numeric

	UnistatNetmonErrors *unistat.Numeric
}

var (
	MS          = 1000.0
	S           = 1000.0 * MS
	timeBuckets = []float64{
		0,
		10 * MS,
		100 * MS,
		200 * MS,
		300 * MS,
		400 * MS,
		500 * MS,
		600 * MS,
		700 * MS,
		800 * MS,
		900 * MS,
		900 * MS,
		1 * S,
		2 * S,
		3 * S,
		4 * S,
		5 * S,
		6 * S,
		7 * S,
		8 * S,
		9 * S,
		10 * S,
		11 * S,
		12 * S,
		13 * S,
		14 * S,
		15 * S,
		30 * S,
		60 * S,
	}
)

func (s *ResolverServer) Register(server *grpc.Server) {
	pb.RegisterTResolverServer(server, s)

	s.UnistatRequests = unistat.NewNumeric("requests", 10, aggr.Counter(), unistat.Sum)
	s.UnistatRequestLatency = unistat.NewHistogram("request_latency", 10, aggr.Histogram(), timeBuckets)

	s.UnistatIssCalls = unistat.NewNumeric("iss_calls", 10, aggr.Counter(), unistat.Sum)
	s.UnistatIssLatency = unistat.NewHistogram("iss_latency", 10, aggr.Histogram(), timeBuckets)
	s.UnistatIssErrors = unistat.NewNumeric("iss_errors", 10, aggr.Counter(), unistat.Sum)

	s.UnistatYpCalls = unistat.NewNumeric("yp_calls", 10, aggr.Counter(), unistat.Sum)
	s.UnistatYpLatency = unistat.NewHistogram("yp_latency", 10, aggr.Histogram(), timeBuckets)
	s.UnistatYpErrors = unistat.NewNumeric("yp_errors", 10, aggr.Counter(), unistat.Sum)

	s.UnistatNannyCalls = unistat.NewNumeric("nanny_calls", 10, aggr.Counter(), unistat.Sum)
	s.UnistatNannyLatency = unistat.NewHistogram("nanny_latency", 10, aggr.Histogram(), timeBuckets)
	s.UnistatNannyErrors = unistat.NewNumeric("nanny_errors", 10, aggr.Counter(), unistat.Sum)

	s.UnistatQypCalls = unistat.NewNumeric("qyp_calls", 10, aggr.Counter(), unistat.Sum)
	s.UnistatQypLatency = unistat.NewHistogram("qyp_latency", 10, aggr.Histogram(), timeBuckets)
	s.UnistatQypErrors = unistat.NewNumeric("qyp_errors", 10, aggr.Counter(), unistat.Sum)

	s.UnistatQloudCalls = unistat.NewNumeric("qloud_calls", 10, aggr.Counter(), unistat.Sum)
	s.UnistatQloudLatency = unistat.NewHistogram("qloud_latency", 10, aggr.Histogram(), timeBuckets)
	s.UnistatQloudErrors = unistat.NewNumeric("qloud_errors", 10, aggr.Counter(), unistat.Sum)

	s.UnistatDeployCalls = unistat.NewNumeric("deploy_calls", 10, aggr.Counter(), unistat.Sum)
	s.UnistatDeployLatency = unistat.NewHistogram("deploy_latency", 10, aggr.Histogram(), timeBuckets)
	s.UnistatDeployErrors = unistat.NewNumeric("deploy_errors", 10, aggr.Counter(), unistat.Sum)

	s.UnistatStaffCalls = unistat.NewNumeric("staff_calls", 10, aggr.Counter(), unistat.Sum)
	s.UnistatStaffLatency = unistat.NewHistogram("staff_latency", 10, aggr.Histogram(), timeBuckets)
	s.UnistatStaffErrors = unistat.NewNumeric("staff_errors", 10, aggr.Counter(), unistat.Sum)

	s.UnistatWalleCalls = unistat.NewNumeric("walle_calls", 10, aggr.Counter(), unistat.Sum)
	s.UnistatWalleLatency = unistat.NewHistogram("walle_latency", 10, aggr.Histogram(), timeBuckets)
	s.UnistatWalleErrors = unistat.NewNumeric("walle_errors", 10, aggr.Counter(), unistat.Sum)

	s.UnistatNetmonErrors = unistat.NewNumeric("netmon_errors", 10, aggr.Counter(), unistat.Sum)

	unistat.Register(s.UnistatRequests)
	unistat.Register(s.UnistatRequestLatency)

	unistat.Register(s.UnistatIssCalls)
	unistat.Register(s.UnistatIssLatency)
	unistat.Register(s.UnistatIssErrors)

	unistat.Register(s.UnistatYpCalls)
	unistat.Register(s.UnistatYpLatency)
	unistat.Register(s.UnistatYpErrors)

	unistat.Register(s.UnistatNannyCalls)
	unistat.Register(s.UnistatNannyLatency)
	unistat.Register(s.UnistatNannyErrors)

	unistat.Register(s.UnistatQypCalls)
	unistat.Register(s.UnistatQypLatency)
	unistat.Register(s.UnistatQypErrors)

	unistat.Register(s.UnistatQloudCalls)
	unistat.Register(s.UnistatQloudLatency)
	unistat.Register(s.UnistatQloudErrors)

	unistat.Register(s.UnistatDeployCalls)
	unistat.Register(s.UnistatDeployLatency)
	unistat.Register(s.UnistatDeployErrors)

	unistat.Register(s.UnistatStaffCalls)
	unistat.Register(s.UnistatStaffLatency)
	unistat.Register(s.UnistatStaffErrors)

	unistat.Register(s.UnistatWalleCalls)
	unistat.Register(s.UnistatWalleLatency)
	unistat.Register(s.UnistatWalleErrors)

	unistat.Register(s.UnistatNetmonErrors)
}

func (s *ResolverServer) collectInstances(ctx context.Context, hostsInfo []netmon.NetmonHostInfo, fieldMask *field_mask.FieldMask) (reply *pb.TInstancesReply, err error) {
	collector := NewInstanceCollector(ctx, s, hostsInfo, fieldMask)
	collector.Spawn()
	reply, err = collector.GetInstancesReply()
	return
}

func (s *ResolverServer) Ping(ctx context.Context, in *pb.TPingRequest) (*pb.TPingReply, error) {
	return &pb.TPingReply{Status: "OK"}, nil
}

func (s *ResolverServer) GetHostInstances(ctx context.Context, in *pb.THostInstancesRequest) (reply *pb.TInstancesReply, err error) {
	defer unistat.MeasureMicrosecondsSince(s.UnistatRequestLatency, time.Now())
	defer s.UnistatRequests.Update(1)

	hostInfo, err := s.WalleClient.GetNetmonHostInfo(in.HostName)
	if err != nil {
		//nolint:S1020
		if _, ok := err.(*netmon.InvalidHostError); ok {
			st := status.New(codes.NotFound, err.Error())
			err = st.Err()
			return
		} else {
			defer s.UnistatNetmonErrors.Update(1)
			log.Logger.Error("can't fetch host info from netmon", aLog.String("host", in.HostName), aLog.Error(err))
			return
		}
	}

	reply, err = s.collectInstances(ctx, []netmon.NetmonHostInfo{*hostInfo}, in.FieldMask)
	return
}

func (s *ResolverServer) GetSwitchInstances(ctx context.Context, in *pb.TSwitchInstancesRequest) (reply *pb.TInstancesReply, err error) {
	defer unistat.MeasureMicrosecondsSince(s.UnistatRequestLatency, time.Now())
	defer s.UnistatRequests.Update(1)

	hosts, err := s.NetmonClient.GetHostsInSwitch(ctx, in.SwitchName)
	if err != nil {
		defer s.UnistatNetmonErrors.Update(1)
		log.Logger.Error("can't fetch switch info from netmon", aLog.String("switch", in.SwitchName), aLog.Error(err))
		return
	}

	hostsInfo, err := s.NetmonClient.GetHostsInfo(ctx, hosts)
	if err != nil {
		defer s.UnistatNetmonErrors.Update(1)
		log.Logger.Error("can't fetch hosts info from netmon", aLog.String("switch", in.SwitchName), aLog.Error(err))
		return
	}

	if len(hosts) != len(hostsInfo) {
		err = fmt.Errorf("not all hosts from switch exists")
		return
	}

	reply, err = s.collectInstances(ctx, hostsInfo, in.FieldMask)
	return
}
