package server

import (
	"io/ioutil"
	"net/http"

	"github.com/grpc-ecosystem/grpc-gateway/runtime"

	aLog "a.yandex-team.ru/library/go/core/log"

	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/log"
)

const (
	swaggerContent = `<!-- HTML for static distribution bundle build -->
<!DOCTYPE html>
<html lang="en">
  <head>
    <meta charset="UTF-8">
    <title>API documentation</title>
    <link rel="stylesheet" type="text/css" href="//unpkg.com/swagger-ui-dist@3/swagger-ui.css" >
    <style>
      html
      {
        box-sizing: border-box;
        overflow: -moz-scrollbars-vertical;
        overflow-y: scroll;
      }
      *,
      *:before,
      *:after
      {
        box-sizing: inherit;
      }
      body
      {
        margin:0;
        background: #fafafa;
      }
    </style>
  </head>

  <body>
    <div id="swagger-ui"></div>
    <script src="//unpkg.com/swagger-ui-dist@3/swagger-ui-bundle.js"></script>
    <script>
    window.onload = function() {
      // Begin Swagger UI call region
      const ui = SwaggerUIBundle({
        url: "/swagger.json",
        dom_id: '#swagger-ui',
        deepLinking: true,
        presets: [
          SwaggerUIBundle.presets.apis,
        ],
        plugins: [
          SwaggerUIBundle.plugins.DownloadUrl
        ]
      })
      // End Swagger UI call region
      window.ui = ui
    }
  </script>
  </body>
</html>
`
)

func NewSwagUI(specPath string, mux *runtime.ServeMux) {
	mux.Handle(
		"GET",
		runtime.MustPattern(runtime.NewPattern(1, []int{2, 0}, []string{""}, "")),
		func(w http.ResponseWriter, r *http.Request, pathParams map[string]string) {
			w.Header().Set("Content-Type", "text/html; charset=utf-8")
			if _, err := w.Write([]byte(swaggerContent)); err != nil {
				log.Logger.Error("unable to write", aLog.Error(err))
				return
			}
		},
	)

	mux.Handle(
		"GET",
		runtime.MustPattern(runtime.NewPattern(1, []int{2, 0}, []string{"swagger.json"}, "")),
		func(w http.ResponseWriter, r *http.Request, pathParams map[string]string) {
			b, err := ioutil.ReadFile(specPath)
			if err != nil {
				w.WriteHeader(http.StatusInternalServerError)
				log.Logger.Error("Unable to read swagger file", aLog.Error(err))
				return
			}

			w.Header().Set("Content-Type", "application/json")
			if _, err := w.Write(b); err != nil {
				log.Logger.Error("unable to write", aLog.Error(err))
				return
			}
		},
	)
}
