package abc

import (
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"net/http"
	"time"

	"golang.org/x/net/context"

	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/log"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/util"
	aLog "a.yandex-team.ru/library/go/core/log"
)

type AbcPerson struct {
	Login string `json:"login"`
}

type AbcService struct {
	ID int64 `json:"id"`
}

type AbcMember struct {
	ID      int64      `json:"id"`
	Person  AbcPerson  `json:"person"`
	Service AbcService `json:"service"`
}

type AbcClient struct {
	client *http.Client
	token  string
	ctx    context.Context
	cancel context.CancelFunc
}

func NewAbcClient(token string) (client *AbcClient, err error) {
	ctx := context.Background()
	ctx, cancel := context.WithCancel(ctx)
	client = new(AbcClient)
	client.client = &http.Client{Timeout: time.Duration(60 * time.Second)}
	client.token = token
	client.ctx = ctx
	client.cancel = cancel
	return
}

func (c *AbcClient) makeRequest(path string) (resp *http.Response, err error) {
	req, err := http.NewRequest("GET", fmt.Sprintf(`https://abc-back.yandex-team.ru%s`, path), nil)
	if err != nil {
		return
	}

	req.Header.Set("Accept", "application/json")
	req.Header.Set("Authorization", fmt.Sprintf(`OAuth %s`, c.token))
	resp, err = c.client.Do(req)
	return
}

func (c *AbcClient) fetchMembers(afterID int64, cb func(AbcMember)) error {
	url := fmt.Sprintf("/api/v4/services/members/?cursor=%d-0&ordering=id&fields=id,person.login,service.id", afterID)
	resp, err := c.makeRequest(url)
	if err != nil {
		return err
	}
	if resp.StatusCode != 200 {
		return fmt.Errorf("request to abc failed with code %d", resp.StatusCode)
	}

	defer func() { _ = resp.Body.Close() }()

	dec := json.NewDecoder(resp.Body)
	for {
		t, err := dec.Token()
		if err == io.EOF {
			break
		}
		if err != nil {
			return err
		}
		switch v := t.(type) {
		case string:
			if v == "results" {
				goto ResultFound
			}
		default:
		}
	}

ResultFound:
	_, err = dec.Token()
	if err != nil {
		return err
	}

	for dec.More() {
		member := AbcMember{}
		err = dec.Decode(&member)
		if err != nil {
			return err
		}

		cb(member)
	}

	for {
		_, err := dec.Token()
		if err == io.EOF {
			break
		}
		if err != nil {
			return err
		}
	}

	return nil
}

func (c *AbcClient) fetchMembersWithRetry(afterID int64, cb func(AbcMember)) error {
	// TODO: request only new records
	var err error

	for retry := 1; retry <= 3; retry++ {
		err = c.fetchMembers(afterID, cb)
		if err == nil {
			break
		}

		sleepDuration := util.RetryBackoff(retry, time.Second, time.Minute)
		select {
		case <-c.ctx.Done():
			return errors.New("abc cache update cancelled")
		case <-time.After(sleepDuration):
			continue
		}
	}

	return err
}

func (c *AbcClient) FetchAllMembers(cb func(AbcMember)) error {
	var lastID, prevID int64
	lastID = 0
	prevID = 0

	wrappedCb := func(member AbcMember) {
		cb(member)
		lastID = member.ID
	}

	err := c.fetchMembersWithRetry(lastID, wrappedCb)
	for err == nil && lastID != prevID {
		prevID = lastID
		log.Logger.Debug("fetching abc members", aLog.Int64("lastId", lastID))
		err = c.fetchMembers(lastID, wrappedCb)
	}
	if err != nil {
		return err
	}

	return nil
}
