package auth

import (
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"net"
	"net/http"
	"net/url"
	"strings"
	"time"

	"golang.org/x/xerrors"
)

type AuthClient struct {
	client *http.Client
	token  string
	port   string
}

func NewAuthClient(token string, port string) (client *AuthClient) {
	client = new(AuthClient)
	client.client = &http.Client{Timeout: time.Duration(5 * time.Second)}
	client.token = token
	client.port = port
	return
}

func (c *AuthClient) makeTvmRequest(ctx context.Context, path string, serviceTicket string) (string, error) {
	req, err := http.NewRequest("GET", fmt.Sprintf("http://localhost:%s%s", c.port, path), nil)
	if err != nil {
		return "", xerrors.Errorf("tvm: %w", err)
	}

	req.Header.Set("Authorization", c.token)
	if serviceTicket != "" {
		req.Header.Set("X-Ya-Service-Ticket", serviceTicket)
	}

	req = req.WithContext(ctx)
	rsp, err := c.client.Do(req)
	if err != nil {
		return "", xerrors.Errorf("tvm: %w", err)
	}

	defer func() { _ = rsp.Body.Close() }()
	body, err := ioutil.ReadAll(rsp.Body)

	if rsp.StatusCode != http.StatusOK {
		if err != nil {
			return "", fmt.Errorf("tvm: tickets: [%d] %s", rsp.StatusCode, rsp.Status)
		}

		return "", fmt.Errorf("tvm: tickets: [%d] %s: %s", rsp.StatusCode, rsp.Status, strings.TrimSpace(string(body)))
	}

	return string(body), nil
}

func (c *AuthClient) getServiceTicket(ctx context.Context) (string, error) {
	// source token can be obtained from https://abc.yandex-team.ru/services/rtc_instance_resolver/resources/
	// dst token found from https://abc.yandex-team.ru/services/passp/resources/
	body, err := c.makeTvmRequest(ctx, "/tvm/tickets?src=2013412&dsts=223", "")
	if err != nil {
		return "", err
	}

	var tickets struct {
		Blackbox struct {
			Ticket string  `json:"ticket"`
			ID     uint32  `json:"tvm_id"`
			Error  *string `json:"error"`
		} `json:"blackbox"`
	}

	dec := json.NewDecoder(strings.NewReader(body))
	if err := dec.Decode(&tickets); err != nil {
		return "", fmt.Errorf("tvm: tickets: invalid json: %v", err)
	}

	if tickets.Blackbox.Error != nil {
		return "", fmt.Errorf("tvm: tickets: %s", strings.TrimSpace(*tickets.Blackbox.Error))
	}

	if tickets.Blackbox.Ticket == "" {
		return "", fmt.Errorf("tvm: tickets: missing ticket")
	}

	return tickets.Blackbox.Ticket, nil
}

func (c *AuthClient) checkServiceTicket(ctx context.Context, serviceTicket string) error {
	body, err := c.makeTvmRequest(ctx, "/tvm/checksrv", serviceTicket)
	if err != nil {
		return err
	}

	var ticket struct {
		Src uint32 `json:"src"`
		Dst uint32 `json:"dst"`
	}

	dec := json.NewDecoder(strings.NewReader(body))
	if err := dec.Decode(&ticket); err != nil {
		return fmt.Errorf("tvm: tickets: invalid json: %v", err)
	}

	return nil
}

// Known errors
var (
	errInvalidSessionID = xerrors.New("invalid Session_id")
	errNeedReset        = xerrors.New("need reset")
)

const (
	statusValid     = "VALID"
	statusNeedReset = "NEED_RESET"
	statusExpired   = "EXPIRED"
	statusNoAuth    = "NOAUTH"
	statusInvalid   = "INVALID"
)

type status struct {
	ID    int    `json:"id"`
	Value string `json:"value"`
}

const (
	intranetURL = "http://blackbox.yandex-team.ru"
)

type displayName struct {
	Name string `json:"name"`
}

type userInfo struct {
	DBFields    map[string]string
	DisplayName displayName
	Login       string
	UID         string
	Scope       string
}

type uid struct {
	Value string `json:"value"`
}

func (c *AuthClient) getUserInfo(ctx context.Context, tvmToken, sessionID, userip string, dbfields []string) (userInfo, error) {
	if sessionID == "" {
		return userInfo{}, errInvalidSessionID
	}

	params := url.Values{
		"method":    {"sessionid"},
		"sessionid": {sessionID},
		"userip":    {userip},
		"host":      {"godoc.qloud.yandex-team.ru"},
		"format":    {"json"},
	}

	if len(dbfields) > 0 {
		params["dbfields"] = []string{strings.Join(dbfields, ",")}
	}

	req, err := http.NewRequest("GET", "https://blackbox.yandex-team.ru/blackbox", nil)
	if err != nil {
		return userInfo{}, err
	}
	req.URL.RawQuery = params.Encode()
	req.Header.Set("X-Ya-Service-Ticket", tvmToken)

	req = req.WithContext(ctx)

	rsp, err := c.client.Do(req)
	if err != nil {
		if urlerr, ok := err.(*url.Error); ok {
			// remove url with sessionid from error message
			err = urlerr.Err
		}

		return userInfo{}, err
	}
	defer func() { _ = rsp.Body.Close() }()

	var data struct {
		Age         int               `json:"age"`
		DBFields    map[string]string `json:"dbfields"`
		DisplayName displayName       `json:"display_name"`
		Error       string            `json:"error"`
		Login       string            `json:"login"`
		Status      status            `json:"status"`
		TTL         string            `json:"ttl"`
		UID         uid               `json:"uid"`
	}
	err = json.NewDecoder(rsp.Body).Decode(&data)
	if err != nil {
		return userInfo{}, xerrors.Errorf("invalid json: %w", err)
	}

	// https://wiki.yandex-team.ru/passport/mda#opisanieprotokola
	switch data.Status.Value {
	case statusValid:
		return userInfo{
			DBFields:    data.DBFields,
			DisplayName: data.DisplayName,
			Login:       data.Login,
			UID:         data.UID.Value,
		}, nil
	case statusNeedReset:
		return userInfo{}, errNeedReset
	case statusExpired:
	case statusNoAuth:
	case statusInvalid:
		return userInfo{}, fmt.Errorf("bad status %s: %s", data.Status.Value, data.Error)
	}

	return userInfo{}, fmt.Errorf("%s: %s", data.Status.Value, data.Error)
}

func (c *AuthClient) CheckAuth(ctx context.Context, r *http.Request) (string, bool) {
	if c.token == "" {
		return "", true
	}

	if serviceTicket := r.Header.Get("X-Ya-Service-Ticket"); serviceTicket != "" {
		err := c.checkServiceTicket(ctx, serviceTicket)
		if err != nil {
			log.Printf("tvm checksrv error: %v", err)
			return "", false
		} else {
			return "", true
		}
	}

	sessionID, err := r.Cookie("Session_id")
	if err != nil {
		return "", false
	}

	tvmToken, err := c.getServiceTicket(ctx)
	if err != nil {
		log.Printf("tvm tickets error: %v", err)
		return "", false
	}

	host, _, err := net.SplitHostPort(r.RemoteAddr)
	if err != nil {
		log.Printf("remote addr split host port error: %v", err)
		return "", false
	}
	info, err := c.getUserInfo(ctx, tvmToken, sessionID.Value, host, nil)
	if err != nil {
		log.Printf("blackbox error: %v", err)
		return "", false
	}

	return info.Login, true
}
