package bot

import (
	"encoding/gob"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"os"
	"sync"
	"time"

	"golang.org/x/net/context"

	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/util"
)

var botTimeToLive, _ = time.ParseDuration("1h")

const botCacheVersion = 1

type BotHostInfo struct {
	InventoryNumber string `json:"Inv"`
	Fqdn            string `json:"FQDN"`
	PlannerID       string `json:"planner_id"`
}

type BotHostMap struct {
	Hosts     map[string]BotHostInfo
	Timestamp time.Time
	Version   int64
}

func (m *BotHostMap) IsActual() bool {
	return time.Now().Before(m.Timestamp.Add(botTimeToLive))
}

func (m *BotHostMap) IsCompatible() bool {
	return m.Version == botCacheVersion
}

type BotClient struct {
	client  *http.Client
	ctx     context.Context
	cancel  context.CancelFunc
	hostMap *BotHostMap
	mutex   sync.Mutex
}

func NewBotClient() (client *BotClient) {
	ctx := context.Background()
	ctx, cancel := context.WithCancel(ctx)
	client = &BotClient{
		client:  &http.Client{Timeout: time.Duration(300 * time.Second)},
		ctx:     ctx,
		cancel:  cancel,
		hostMap: nil,
	}
	return
}

func (c *BotClient) makeRequest(path string) (resp *http.Response, err error) {
	req, err := http.NewRequest("GET", fmt.Sprintf(`https://bot.yandex-team.ru%s`, path), nil)
	if err != nil {
		return
	}

	req.Header.Set("Accept", "application/json")
	req = req.WithContext(c.ctx)
	resp, err = c.client.Do(req)
	return
}

func (c *BotClient) dumpHostMap(hostMap *BotHostMap, filePath string) error {
	temporaryPath := fmt.Sprintf("%s.tmp", filePath)
	temporaryFile, err := os.Create(temporaryPath)
	if err != nil {
		return err
	}

	enc := gob.NewEncoder(temporaryFile)
	err = enc.Encode(*hostMap)
	if err != nil {
		return err
	}

	err = temporaryFile.Close()
	if err != nil {
		return err
	}

	return os.Rename(temporaryPath, filePath)
}

func (c *BotClient) loadHostMap(p string) (hostMap *BotHostMap, err error) {
	cacheFile, err := os.Open(p)
	if err != nil {
		return
	}
	defer func() { _ = cacheFile.Close() }()

	hostMap = new(BotHostMap)
	dec := gob.NewDecoder(cacheFile)
	err = dec.Decode(hostMap)
	if err != nil {
		return
	}

	if !hostMap.IsCompatible() {
		err = errors.New("bot host map cache isn't compatible")
	}

	return
}

func (c *BotClient) fetchHostMap() (result *BotHostMap, err error) {
	result = &BotHostMap{
		Hosts:     make(map[string]BotHostInfo),
		Timestamp: time.Now(),
		Version:   botCacheVersion,
	}
	resp, err := c.makeRequest("/api/view.php?name=view_oops_hardware&format=json")
	if err != nil {
		return
	}

	defer func() { _ = resp.Body.Close() }()

	dec := json.NewDecoder(resp.Body)

	_, err = dec.Token()
	if err != nil {
		return
	}

	for dec.More() {
		var hostInfo struct {
			InventoryNumber *string `json:"Inv"`
			Fqdn            *string `json:"FQDN"`
			PlannerID       *string `json:"planner_id"`
		}
		err = dec.Decode(&hostInfo)
		if err != nil {
			return
		}

		if hostInfo.Fqdn != nil && hostInfo.InventoryNumber != nil && hostInfo.PlannerID != nil {
			result.Hosts[*hostInfo.Fqdn] = BotHostInfo{
				InventoryNumber: *hostInfo.InventoryNumber,
				Fqdn:            *hostInfo.Fqdn,
				PlannerID:       *hostInfo.PlannerID,
			}
		}
	}

	_, err = dec.Token()
	if err != nil {
		return
	}

	return
}

func (c *BotClient) updateHostMap() (int, error) {
	var hostMap *BotHostMap
	var err error

	for retry := 1; retry <= 3; retry++ {
		hostMap, err = c.fetchHostMap()
		if err == nil {
			break
		}

		sleepDuration := util.RetryBackoff(retry, time.Second, time.Minute)
		select {
		case <-c.ctx.Done():
			return 0, errors.New("bot host map update cancelled")
		case <-time.After(sleepDuration):
			continue
		}
	}
	if err != nil {
		return 0, err
	}

	c.mutex.Lock()
	c.hostMap = hostMap
	c.mutex.Unlock()

	return len(hostMap.Hosts), nil
}

func (c *BotClient) getActualHostMap() (result *BotHostMap) {
	c.mutex.Lock()
	result = c.hostMap
	c.mutex.Unlock()
	if result != nil && !result.IsActual() {
		result = nil
	}
	return
}

func (c *BotClient) GetHostInfo(fqdn string) (result *BotHostInfo, err error) {
	hostMap := c.getActualHostMap()
	if hostMap == nil {
		err = errors.New("bot host map is outdated")
		return
	}

	if val, ok := hostMap.Hosts[fqdn]; ok {
		result = &val
	} else {
		result = nil
	}
	return
}
