package golovan

import (
	"encoding/gob"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"os"
	"sync"
	"time"

	"golang.org/x/net/context"

	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/util"
)

var golovanTimeToLive, _ = time.ParseDuration("1h")

const golovanCacheVersion = 1

type InvalidItypeError struct {
}

type GolovanItypeInfo struct {
	Tags []string
}

type GolovanItypeMap struct {
	Itypes    map[string]GolovanItypeInfo
	Timestamp time.Time
	Version   int64
}

func (m *GolovanItypeMap) IsActual() bool {
	return time.Now().Before(m.Timestamp.Add(golovanTimeToLive))
}

func (m *GolovanItypeMap) IsCompatible() bool {
	return m.Version == golovanCacheVersion
}

type GolovanClient struct {
	client   *http.Client
	ctx      context.Context
	cancel   context.CancelFunc
	itypeMap *GolovanItypeMap
	mutex    sync.Mutex
}

func NewGolovanClient() (client *GolovanClient) {
	ctx := context.Background()
	ctx, cancel := context.WithCancel(ctx)
	client = &GolovanClient{
		client:   &http.Client{Timeout: time.Duration(30 * time.Second)},
		ctx:      ctx,
		cancel:   cancel,
		itypeMap: nil,
	}
	return
}

func (c *GolovanClient) makeRequest(path string) (resp *http.Response, err error) {
	req, err := http.NewRequest("GET", fmt.Sprintf(`https://yasm.yandex-team.ru%s`, path), nil)
	if err != nil {
		return
	}

	req.Header.Set("Accept", "application/json")
	req = req.WithContext(c.ctx)
	resp, err = c.client.Do(req)
	return
}

func (c *GolovanClient) dumpItypeMap(itypeMap *GolovanItypeMap, filePath string) error {
	temporaryPath := fmt.Sprintf("%s.tmp", filePath)
	temporaryFile, err := os.Create(temporaryPath)
	if err != nil {
		return err
	}

	enc := gob.NewEncoder(temporaryFile)
	err = enc.Encode(*itypeMap)
	if err != nil {
		return err
	}

	err = temporaryFile.Close()
	if err != nil {
		return err
	}

	return os.Rename(temporaryPath, filePath)
}

func (c *GolovanClient) loadItypeMap(p string) (itypeMap *GolovanItypeMap, err error) {
	cacheFile, err := os.Open(p)
	if err != nil {
		return
	}
	defer func() { _ = cacheFile.Close() }()

	itypeMap = new(GolovanItypeMap)
	dec := gob.NewDecoder(cacheFile)
	err = dec.Decode(itypeMap)
	if err != nil {
		return
	}

	if !itypeMap.IsCompatible() {
		err = errors.New("golovan itype map cache isn't compatible")
	}

	return
}

func (c *GolovanClient) fetchItypeMap() (result *GolovanItypeMap, err error) {
	result = &GolovanItypeMap{
		Itypes:    make(map[string]GolovanItypeInfo),
		Timestamp: time.Now(),
		Version:   golovanCacheVersion,
	}
	resp, err := c.makeRequest("/metainfo/tagkeys")
	if err != nil {
		return
	}
	if resp.StatusCode != 200 {
		return nil, fmt.Errorf("request to golovan failed with code %d", resp.StatusCode)
	}

	defer func() { _ = resp.Body.Close() }()

	dec := json.NewDecoder(resp.Body)

	_, err = dec.Token()
	if err != nil {
		return
	}

	for dec.More() {
		var itype json.Token
		itype, err = dec.Token()
		if err != nil {
			return
		}

		var tags []string
		err = dec.Decode(&tags)
		if err != nil {
			return
		}

		result.Itypes[itype.(string)] = GolovanItypeInfo{
			Tags: tags,
		}
	}

	_, err = dec.Token()
	if err != nil {
		return
	}

	return
}

func (c *GolovanClient) updateItypeMap() (int, error) {
	var itypeMap *GolovanItypeMap
	var err error

	for retry := 1; retry <= 3; retry++ {
		itypeMap, err = c.fetchItypeMap()
		if err == nil {
			break
		}

		sleepDuration := util.RetryBackoff(retry, time.Second, time.Minute)
		select {
		case <-c.ctx.Done():
			return 0, errors.New("golovan itype map update cancelled")
		case <-time.After(sleepDuration):
			continue
		}
	}
	if err != nil {
		return 0, err
	}

	c.mutex.Lock()
	c.itypeMap = itypeMap
	c.mutex.Unlock()

	return len(itypeMap.Itypes), nil
}

func (c *GolovanClient) getActualItypeMap() (result *GolovanItypeMap) {
	c.mutex.Lock()
	result = c.itypeMap
	c.mutex.Unlock()
	if result != nil && !result.IsActual() {
		result = nil
	}
	return
}

func (c *GolovanClient) GetItypeInfo(itype string) (result *GolovanItypeInfo, err error) {
	itypeMap := c.getActualItypeMap()
	if itypeMap == nil {
		err = errors.New("golovan itype map is outdated")
		return
	}

	if val, ok := itypeMap.Itypes[itype]; ok {
		result = &val
	} else {
		result = nil
	}
	return
}
