namespace java ru.yandex.iss.thrift
namespace py iss_thrift3

/* COMMON : BEGIN*/
/* Just a description useful for write-operations */
struct AuthorSignature {
    /* Author identifier (free form) */
    10: required string author,
    /* Change description (free form) */
    20: required string message = ""
}

/* Configuration Identifier in <pre>family#randomid</pre> format */
typedef string ConfigurationId

/* Instance identifier (unique only in Configuration scope) also known as Slot. Format:  <pre>service_id@host.fqdn.tld</pre> */
typedef string Slot

/* Full instance identifier: Slot + Configuration identifier */
struct InstanceId {
    /* slot definition. Format is <pre>service_id@host.fqdn.tld</pre> */
    10: required Slot slot,
    /* Configuration id as String with format family#contenthash*/
    20: required ConfigurationId configuration
}

/* Representation of unbound instance (not related to any configuration) */
struct Instance {
    /* slot definition. Format is <pre>service_id@host.fqdn.tld</pre> */
    10: required Slot slot,
    /* Instance properties (free form) */
    20: required map<string, string> properties,
    /* instance in text form */
    30: optional string textProperties
}

/* Representation of bound instance (Instance of concrete Configuration) */
struct InstanceProperties {
    /* Instance identifier */
    10: required InstanceId id,
    30: required map<string, string> properties,
    40: optional string textProperties
}

struct Version {
    /** Version from artifactory */
    10: required string version,
    /** Compilation timestamp */
    20: required string timestamp
}

/* Identifier of Cacher instance request was handled by */
struct ServiceSignature {
    10: required string hostFQDN
}

/* Offset+limit for requests returning long lists of results */
struct Portion {
    10: i32 offset = 0,
    20: i32 limit = 100
}

/* Just a timeframe with begin and end in unix timestamp format (in milliseconds) */
struct TimeFrame {
    10: i64 beginTimestamp = -1,
    20: i64 endTimestamp = -1,
}

struct SecurityToken {
    10: required string user,
    // timestamp of request
    20: required i64 timestamp,
    // request hash with digital signature TODO: wiki
    30: required binary digest
}

enum ConsistencyLevel {
    /* rely on keyspace/column family defaults */
    DEFAULT = 0,

    /* ack only from single replica */
    ONE = 10,

    /* ack from quorum of all replicas */
    QUORUM = 20,

    /* ack from quorum of replicas from current DC */
    LOCAL_QUORUM = 30,

    /* ack from all quorums in each DC. If some DC is unavailable, then this request will fail */
    EACH_QUORUM = 40,

    /* ack from all replicas, datacenter unaware*/
    ALL = 50,

    /* no ack from replicas, if no replicas available for the record, it will be written into hinted handoff. Useful for high-available writes*/
    ANY = 60,

    /* ack from two replicas, datacenter unaware  */
    TWO = 70,

    /* ask from three replicas, datacenter unaware */
    THREE = 80,
}
/* COMMON : END*/

/* EXCEPTIONS : BEGIN*/
/* Error code for SystemException */
enum ErrorCode {
    /* Not implemented yet */
    NOT_IMPLEMENTED = 1,

    /** Cassandra DB is not accessible */
    CASSANDRA_CONNECTION_BROKEN = 2,

    /* NPE and other unpredictable behavior causes this error code */
    INTERNAL_ERROR = 4,

    /* feature was disabled, user can enable or disable features */
    FEATURE_DISABLED_EXCEPTION = 5,

    /* Consistency cannot be achieved */
    CANNOT_ACHIEVE_CONSISTENCY = 6
}

/* This exception will be thrown when incorrect parameters was passed to some method */
exception InvalidRequestException {
    10: required string message,
    20: string stacktrace,
    /* Identifier of cacher exception happened on */
    30: required ServiceSignature signature
}

/* Exception to be thrown when some entity was not found by Cacher (configuration, branch, etc) */
exception NotFoundException {
    10: required string message,
    20: string stacktrace,
    /* Identifier of cacher exception happened on */
    30: required ServiceSignature signature
}

/* Exception to be thrown when some entity exists but is not loaded into cache.
 * It will be loaded asynchronously, so client should retry request. */
exception NotReadyException {
    10: required string message,
    20: string stacktrace,
    /* Identifier of cacher exception happend on */
    30: required ServiceSignature signature
}

/* This exception will be thrown by any failed request without specific handler */
exception SystemException {
    10: required string message,
    20: required ErrorCode errorCode,
    /* Identifier of cacher exception happened on */
    30: required ServiceSignature signature,
    40: string stacktrace
}

/* Exception is throws from getHostConfiguration and signals that footprint on configuration equals and there is no need to change configuration */
exception HostConfigurationUnchangedException {
    10: required ServiceSignature signature
}
/* EXCEPTIONS : END*/

/* UNSTRUCTURED : BEGIN */
/* Configuration meta information */
struct ConfigurationDescription {
    /* Configuration id as String with format family#contenthash*/
    10: required ConfigurationId id,
    /* Parent configuration id. May be not defined when configuration has no parent */
    20: optional ConfigurationId parent,
    /* Configuration creation timestamp (Cacher time) */
    30: required i64 timestamp,
    /* Configuration author signature */
    40: required AuthorSignature signature,
    /* whether this configuration can be applied. */
    50: required bool isApplicable;
    /* arbitrary user properties, set during createConfiguration call */
    60: optional map<string, string> properties;
}

/* Everything required for target state change: query to select instances from some configurations and target state wanted to be applied */
struct TargetStateTransition {
    /* List of instance filters defined as string representations of queries.
     * Required grammar is new one.
     *  Query example: <pre>((configuration.id == "validation#f1bdbb2e") && (!instance[test-prop]))</pre>
     * */
    10: required list<string> filters,

    /* New target state for selected instances. May be REMOVED, PREPARED or ACTIVE */
    20: required string state,
}

struct SlotTargetStateTransition {
    /* map form configurationId to target state. May be REMOVED, PREPARED or ACTIVE */
    10: required map<ConfigurationId, string> configurationTargetState,
    /* old etag for CAS update */
    20: optional string etag
}

/* Agent status report */
struct DetailedCurrentState {
    /* Report acceptance timestamp (Cacher time) */
    10: required i64 timestamp,
    /* Current state reported by agent (free-form) */
    20: required string state,
    /* Some properties related to report (free-form)*/
    30: required map<string, string> properties,
    /* Server time when object was received. Ignored when set by client*/
    40: optional i64 serverTimestamp,
    /* identifier of instance */
    50: optional InstanceId instanceId,
    /* string representation of the properties. This field is filled if format parameter is set to some text format */
    60: optional string textProperties,
}


/* Full instance state: Slot, Configuration Identifier, target state and current state */
struct InstanceState {
    /* Instance identifier */
    10: required InstanceId id,
    /* Target state of instance. May be REMOVED, PREPARED or ACTIVE */
    20: required string targetState,
    /* Current state of instance. Free-form. */
    30: required string feedbackState
}

/* Extended full instance state: Slot, Configuration Identifier, target state and current state and metadata */
struct DetailedInstanceState {
    /* Instance identifier */
    10: required InstanceId id,
    /* Target state value */
    20: required string target,
    /* Agent status report */
    30: required DetailedCurrentState feedback,
}

struct HostConfigurationInstance {
    10: required InstanceId id,
    20: required string targetState,
    30: required map<string, string> properties,
    40: optional string stateOperationId,
    /* timestamp of transition creation */
    50: optional i64 transitionTimestamp,
}

/* Result of filtering instances in some configuration with some query */
struct QueryResult {
    /* Configuration which instances was filtered */
    10: required ConfigurationId configuration,
    /* List of slots of instances which fulfills query criteria */
    20: required list<Slot> instances
}

/* UNSTRUCTURED : END */

/* TARGET_STATE : BEGIN */
typedef string TargetState
typedef string ETag

enum TargetStateUpdateErrorCode {
     ETAG_DID_NOT_MATCH = 1,
     EXECUTION_EXCEPTION = 2,
     TARGET_STATE_CHANGES_PROHIBITED = 3,
     /* unknown internal error  */
     UNKNOWN = 4
}

struct TargetStateUpdateError {
    10: required TargetStateUpdateErrorCode code,
    20: required string description
}

/* Metadata of new configuration and Cacher identifier */
struct CreateConfigurationResponse {
    /* Metadata of new configuration */
    10: required ConfigurationDescription configuration,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature
}

/* Metadata of new configuration and Cacher identifier */
struct CopyAndModifyConfigurationFromExternalResponse {
    /* Metadata of new configuration */
    10: required ConfigurationDescription configuration,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature
}

/* Metadata of new configuration and Cacher identifier */
struct CopyAndModifyConfigurationResponse {
    /* Metadata of new configuration */
    10: required ConfigurationDescription configuration,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature
}

/* Just information about cacher identifier */
struct DestroyConfigurationResponse {
    20: required bool destroyed,
    /* Identifier of cacher request was processed on */
    10: required ServiceSignature signature
}

/* Url of exported configuration and Cacher identifier */
struct ExportConfigurationResponse {
    /* URL containing exported configuration */
    10: required string url,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature
}

/* TODO */
struct ValidationMessage {
   /* */
   1: required string type,
   /* */
   2: required string ruleName,
   /* */
   3: required string description,
   /* */
   4: required i64 timestamp
}

/* TODO */
struct ValidationReport {
   /* */
   1: required list<ValidationMessage> messages,
   /* */
   2: required list<string> rules,
   /* */
   3: required map<string,i64> rulesExecutionDuration
}

/* Configuration metadata and Cacher identifier */
struct GetConfigurationDescriptionResponse {
    /* Configuration metadata */
    10: required ConfigurationDescription configuration,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature
}

/* Contains only Cacher identifier. Obtaining instance of this class means request success. */
struct SetBranchHeadResponse {
    /* Identifier of cacher request was processed on */
    10: required ServiceSignature signature
}

/* Contains only Cacher identifier. Obtaining instance of this class means request success. */
struct RemoveBranchHeadResponse {
    /* Identifier of cacher request was processed on */
    10: required ServiceSignature signature
}

/* Configuration identifier of requested branch head and Cacher identifier */
struct GetBranchHeadResponse {
    /* Configuration id as String with format family#contenthash*/
    10: required ConfigurationId configuration,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature
}

/* Set of configuration identifiers of requested branch heads and Cacher identifier */
struct GetBranchHeadsResponse {
    /* Set of configuration identifiers as Strings with format family#contenthash*/
    10: required set<ConfigurationId> configurations,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature
}

/* Per-configuration sets of instances affected by target state change operation */
struct ApplyResponse {
    /* Keys of this map are identifiers of configurations affected by request. Values are sets of slots in this configuration
    * which state was changed*/
    10: required map<ConfigurationId, set<Slot>> affected,

    /* Identifier of cacher request was processed on */
    30: required ServiceSignature signature

    20: required list<ConfigurationId> nonApplicable_OBSOLETE,
}

/* Instance state (target+current) and Cacher identifier */
struct GetInstanceStateResponse {
    /*  Instance state (target and current)*/
    10: required InstanceState instance,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature
}

/* Extended instance state (target+current+metadata) and Cacher identifier */
struct GetDetailedInstanceStateResponse {
    /* Extended instance state (target and current and metadata)*/
    10: required DetailedInstanceState instance,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature
}

/* Instance properties (as string-to-string map)*/
struct GetInstancePropertiesResponse {
    /* Instance properties */
    10: required InstanceProperties instance,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature
}

/* List of per-configuration results of some instances query */
struct QueryResponse {
    /* Per-configuration query results */
    10: required list<QueryResult> result,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature,
}


/* Full configuration representation. */
struct GetConfigurationResponse {
    /* Instances of this configuration. This list may be paged by GetConfigurationResponse method (see method description for details) */
    10: required list<Instance> instances,
    /* Configuration Identifier in <pre>family#randomid</pre> format */
    30: required ConfigurationId id,
    /* Parent Configuration Identifier in <pre>family#randomid</pre> format. May be undefined. */
    40: optional ConfigurationId parentId,
    /* Identifier of cacher request was processed on */
    50: required ServiceSignature signature,

    /* not supported as mutations are not saved now */
    20: required list<string> mutations_OBSOLETE,
}

struct GetConfigurationFamiliesResponse {
    10: required set<string> families,
    20: required ServiceSignature signature,
}

// TODO: remove deprecated 20:
struct GetFamilyConfigurationsResponse {
    10: required set<ConfigurationId> configurations,
    20: required ServiceSignature signature,
}

/* Cacher version */
struct VersionResponse {
    /* Cacher version as string */
    10: required Version version,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature
}

/* Configuration of any concrete hosts. */
struct GetHostConfigurationResponse {
    /* List of all the bound instances (slot + configuration identifier) related to requested host which are in ACTIVE or PREPARED state */
    10: required list<HostConfigurationInstance> instances,

    /* Hash of the configuration. Can be used with getHostConfiguration to reduce network traffic. See comments on getHostConfiguration */
    30: optional string hostConfigurationFootprint,

    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature,
}

/* Configuration of any concrete hosts. */
struct GetSlotConfigurationResponse {
    /* List of all the bound instances (slot + configuration identifier) related to requested host which are in ACTIVE or PREPARED state */
    10: required list<HostConfigurationInstance> instances,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature,
}

/* Configuration of any concrete hosts. */
struct GetSlotsConfigurationResponse {
    /* List of all the bound instances (slot + configuration identifier) related to requested host which are in ACTIVE or PREPARED state */
    10: required map<string, list<HostConfigurationInstance>> instances,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature,
}

/* Contains only Cacher identifier. Obtaining instance of this class means request success. */
struct SaveFeedbackResponse {
    /* Identifier of cacher request was processed on */
    10: required ServiceSignature signature
}

/* Rendered MVEL template and Cacher identifier*/
struct RenderTemplateResponse {
    /* MVEL template rendered in Cacher context */
    10: required string result,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature
}

struct ApplyTargetStateResponse {
    /* Identifier of cacher request was processed on */
    10: required ServiceSignature signature;
    /* slot to etag mapping*/
    20: required map<Slot, ETag> etags;
    /* slot to error mapping */
    30: required map<Slot, TargetStateUpdateError> errors;

}

struct ToggleMethodResponse {
    /* Identifier of cacher request was processed on */
    10: required ServiceSignature signature;
}

struct GetAliveConfigurationsResponse {
    /* Identifier of cacher request was processed on */
    10: required ServiceSignature signature;
    /* set of configurations which are active in some slots */
    20: required set<ConfigurationId> configurations;
}

/* Big content of known format passed directly in arguments */
struct InlineContent {
    /* Any data, its format is described in following fields */
    10: required binary data;
    /* Format of unpacked data - "CMS", "JSON", etc */
    20: required string format;
    /* Data compression method - "DEFLATE" for deflate/gzip compression,
     * "NONE" - no compression, default - same as "NONE" */
    30: optional string compressionMethod;
}

struct BranchHistoryItem {
    /* Configuration to which the branch was set */
    10: required ConfigurationId configuration;
    /* Item author signature */
    20: required AuthorSignature author;
    /* Item creation time */
    30: required i64 timestamp;
}

struct GetBranchHistoryResponse {
    /* Identifier of cacher request was processed on */
    10: required ServiceSignature signature;
    /* History items matching request */
    20: required list<BranchHistoryItem> items;
}

struct GetCurrentStateHistoryResponse {
    /* Identifier of cacher request was processed on */
    10: required ServiceSignature signature;
    /* History items matching request */
    20: required list<DetailedCurrentState> items;
}

struct TargetStateHistoryItem {
    /* Affected configurations and their target states */
    10: required SlotTargetStateTransition transition;
    /* Item author signature */
    20: required AuthorSignature author;
    /* Item creation timestamp */
    30: required i64 timestamp;
}

struct GetTargetStateHistoryResponse {
    /* Identifier of cacher request was processed on */
    10: required ServiceSignature signature;
    /* History items matching request */
    20: required list<TargetStateHistoryItem> items;
}

struct GetClassSchemaResponse {
    /* Identifier of cacher request was processed on */
    10: required ServiceSignature signature;
    /* schema of the class */
    20: required string schema;
}

service AgentService {
    /* Get all the data related to given hostname. Such data must include any instances in PREPARED or ACTIVE state with host part of slot equal to given one.
     This method intended to be used by Agent. */
    GetHostConfigurationResponse
        getHostConfiguration(10: required string hostFQDN,
                            /* GetHostConfigurationResponse may contain configurationFingerprint,
                            if this parameter is passed, then server will throw ConfigurationNotChangedException if current configuration has same footprint.
                            It's done to reduce traffic  */
                            20: string previousHostConfigurationFootprint)
        throws (10:InvalidRequestException ire,
                40:SystemException se,
                60:NotReadyException nre,
                50:HostConfigurationUnchangedException hcue),


    /* Return host configuration with REMOVED instances presented, it is used to seed just removed instances */
    GetHostConfigurationResponse
            getHostConfigurationWithRemoved(10: required string hostFQDN,
                                /* GetHostConfigurationResponse may contain configurationFingerprint,
                                if this parameter is passed, then server will throw ConfigurationNotChangedException if current configuration has same footprint.
                                It's done to reduce traffic  */
                                20: string previousHostConfigurationFootprint)
            throws (10:InvalidRequestException ire,
                    40:SystemException se,
                    60:NotReadyException nre,
                    50:HostConfigurationUnchangedException hcue),

    /* Save current state report. This method intended to be used by Agent. */
    SaveFeedbackResponse
        saveFeedback(10: required InstanceId instance,
                     20: required DetailedCurrentState feedbackMessage)
        throws (10:InvalidRequestException ire,
                40:SystemException se,
                50:NotReadyException nre)
}

service IssService extends AgentService {
    /* create new configuration from dump from external URL */
    CreateConfigurationResponse
        createConfigurationFromExternal(10: required string family,
                                        20: required string dumpURL,
                                        30: required string dumpFormat,
                                        40: required AuthorSignature signature,
                                        50: required bool isApplicable,
                                        60: string configurationId,
                                        70: map<string, string> properties)
        throws (10:InvalidRequestException ire,
                20:SystemException se,
                100:NotReadyException nre),

    /* create configuration from existing by modifying it with filler from external URL */
    CopyAndModifyConfigurationFromExternalResponse
        copyAndModifyConfigurationFromExternal(10: required ConfigurationId parent,
                                               20: required string modifierURL,
                                               30: required string modifierFormat,
                                               40: required AuthorSignature signature,
                                               50: required bool isApplicable,
                                               60: string configurationId,
                                               70: map<string, string> properties)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se,
                100:NotReadyException nre),

    /* create new configuration from dump content */
    CreateConfigurationResponse
        createConfigurationFromContent(10: required string family,
                                       20: required InlineContent dump,
                                       50: required AuthorSignature signature,
                                       60: required bool isApplicable,
                                       70: string configurationId,
                                       80: map<string, string> properties)
        throws (10:InvalidRequestException ire,
                20:SystemException se,
                100:NotReadyException nre),

    /* create configuration from existing by modifying it with filler content */
    CopyAndModifyConfigurationFromExternalResponse
            copyAndModifyConfigurationFromContent(10: required ConfigurationId parent,
                                                  20: required InlineContent modifier,
                                                  50: required AuthorSignature signature,
                                                  60: required bool isApplicable,
                                                  70: string configurationId,
                                                  80: map<string, string> properties)
            throws (10:InvalidRequestException ire,
                    20:NotFoundException nfe,
                    40:SystemException se,
                    100:NotReadyException nre),

    /* create configuration from given instances */
    CreateConfigurationResponse
        createConfiguration(10: required string family
                            20: required list<Instance> instances,
                            30: required AuthorSignature signature,
                            40: required bool isApplicable,
                            50: string configurationId,
                            60: map<string, string> properties)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se,
                50:NotReadyException nre),

    /* create configuration from existing by applying mutation rules */
    CopyAndModifyConfigurationResponse
        copyAndModifyConfiguration(10: required ConfigurationId parent,
                                   20: required string mutation,
                                   30: required AuthorSignature signature,
                                   40: required bool isApplicable,
                                   50: string configurationId,
                                   60: map<string, string> properties)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se,
                50:NotReadyException nre),

    /* Totally remove configuration from ISS. InvalidRequestException will be risen if configuration is prepared or active in some slots */
    DestroyConfigurationResponse
        destroyConfiguration(10: required ConfigurationId configuration,
                             20: required AuthorSignature signature,
                             30: SecurityToken token)
        throws (10:InvalidRequestException ire,
                30:NotFoundException nfe,
                20:SystemException se,
                40:NotReadyException nre),

    GetConfigurationFamiliesResponse
        getConfigurationFamilies(10: optional string pattern)
            throws (10:InvalidRequestException ire,
                    20:SystemException se,
                    30:NotReadyException nre),

    GetFamilyConfigurationsResponse
        getFamilyConfigurations(10: required string family)
            throws (10:InvalidRequestException ire,
                    20:SystemException se,
                    30:NotReadyException nre),

    /* Set given configuration as head of given branch (in configuration's family) */
    SetBranchHeadResponse
        setBranchHead(10: required ConfigurationId configuration,
                      20: required string branch,
                      40: required AuthorSignature signature)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se,
                50:NotReadyException nre),

    /* Unset configuration (if any) set as head of given branch in given family) */
    RemoveBranchHeadResponse
        removeBranchHead(10: required string family,
                         20: required string branch,
                         30: required AuthorSignature signature)
        throws (10:InvalidRequestException ire,
                        20:NotFoundException nfe,
                        40:SystemException se,
                        50:NotReadyException nre),

    /* Apply new target state to set of instances defined by given query */
    ApplyResponse
        apply(10: required TargetStateTransition transition,
              20: required AuthorSignature signature,
              30: ConsistencyLevel consistencyLevel = ConsistencyLevel.DEFAULT,
              40: SecurityToken token)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se,
                50:NotReadyException nre),

    /* Apply new target state to set of instances defined by given query */
    ApplyTargetStateResponse
        applyTargetState(10: required map<Slot, SlotTargetStateTransition> transition,
                20: required AuthorSignature signature,
                30: ConsistencyLevel consistencyLevel = ConsistencyLevel.DEFAULT,
                40: SecurityToken token)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se,
                50:NotReadyException nre),

    /* Renders any MVEL template in context of Cacher API */
    RenderTemplateResponse
        renderTemplate(10: required string template,
                       20: required InstanceId instanceId)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se,
                50:NotReadyException nre)

    /* Load configuration metadata (parent identifier, creation timestamp, etc) */
    GetConfigurationDescriptionResponse
        getConfigurationDescription(10: required ConfigurationId configuration)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se,
                50:NotReadyException nre),

    /* Load all the configuration content. <b>>WARNING</b>: result may contain large amount of data (really large), so method
     is paged and should not be used frequently */
    GetConfigurationResponse
            getConfiguration(10: required ConfigurationId configuration,
                             20: required Portion portion,
                             /* format in which instance properties shall be returned. */
                             30: string dumpFormat)
            throws (10:InvalidRequestException ire,
                    20:NotFoundException nfe,
                    40:SystemException se,
                    50:NotReadyException nre),

    /* Get configuration identifier set as head of given branch in given family */
    GetBranchHeadResponse
        getBranchHead(10: required string family
                      20: required string branch)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se,
                50:NotReadyException nre),

    /* Get all the heads of given branches in all the configuration families */
    GetBranchHeadsResponse
        getBranchHeads(10: required string branch)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se,
                50:NotReadyException nre),

    /* Perform instances query. Returns per-configuration lists of instances. Result is paged. */
    QueryResponse
        query(10: required string filter,
              20: required Portion portion)
        throws (10:InvalidRequestException ire,
                40:SystemException se,
                50:NotReadyException nre),

    /* Get state (target+current) of given instance in given configuration */
    GetInstanceStateResponse
        getInstanceState(10: required InstanceId instanceId)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se,
                50:NotReadyException nre),

    /*Get state of given slot */
    GetSlotConfigurationResponse
        getSingleSlotConfiguration(10: required Slot slot)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se,
                50:NotReadyException nre),

    /* Get state of given slots */
    GetSlotsConfigurationResponse
        getSlotsConfiguration(10: required set<Slot> slots)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se,
                50:NotReadyException nre),

    /* Get extended state (target+current+metadata) of given instance in given configuration */
    GetDetailedInstanceStateResponse
        getDetailedInstanceState(10: required InstanceId instanceId,
                                /* format in which current state properties shall be returned. */
                                20: string dumpFormat)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se,
                50:NotReadyException nre),

    /* Get properties of given instance in given configuration */
    GetInstancePropertiesResponse
        getInstanceProperties(10: required InstanceId instanceId,
                            /* format in which instance properties shall be returned. */
                            20: string dumpFormat)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se,
                50:NotReadyException nre),

    /* Get list of configurations which are active of prepared in some slots */
    GetAliveConfigurationsResponse
        getAliveConfigurations()
        throws (10:SystemException se,
                50:NotReadyException nre),

    /* Cacher version */
    VersionResponse
        getVersion()
        throws (10:SystemException se,
                 50:NotReadyException nre),

    /* Enable/disable some features */
    ToggleMethodResponse
        toggleFeature(10: required string feature,
                     20: required bool enabled,
                     30: required AuthorSignature author,
                     40: ConsistencyLevel consistencyLevel = ConsistencyLevel.DEFAULT,
                     50: SecurityToken token)
        throws (10:InvalidRequestException ire,
                20:SystemException se,
                30:NotReadyException nre),

    /* Get branch history for time frame, both ends included, sorted by timestamp descending */
    GetBranchHistoryResponse
        getBranchHistory(10: required string family,
                         20: required string branch,
                         30: required TimeFrame timeFrame,
                         40: required i32 recordLimit)
        throws (10:InvalidRequestException ire,
                20:SystemException se),

    /* Get current state history for time frame, both ends included, sorted by timestamp descending */
    GetCurrentStateHistoryResponse
        getCurrentStateHistory(10: required Slot slot,
                               20: required TimeFrame timeFrame,
                               30: required i32 recordLimit,
                               /* format in which current state properties shall be returned. */
                               40: string dumpFormat)
        throws (10:InvalidRequestException ire,
                20:SystemException se),

    /* Get current state history for time frame, both ends included, sorted by timestamp descending */
    GetTargetStateHistoryResponse
        getTargetStateHistory(10: required Slot slot,
                              20: required TimeFrame timeFrame,
                              30: required i32 recordLimit)
        throws (10:InvalidRequestException ire,
                20:SystemException se),

    /* get schema of the class */
    GetClassSchemaResponse
        getClassSchema(10: required string className,
                  /* Format for which schema should be returned. Same as dumpFormat in createConfigurationFromExternal */
                  20: required string dumpFormat)
                  /* bad class name or dump format*/
        throws (10:InvalidRequestException ire,
                /* class was not found */
                20:NotFoundException nfe,
                30:SystemException se),
}
