package iss3

import (
	"crypto/tls"
	"fmt"
	"net"
	"time"

	"golang.org/x/net/context"

	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/lru"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/util"
	"git.apache.org/thrift.git/lib/go/thrift"
)

var issTimeToLive, _ = time.ParseDuration("1m")

const issCacheEntries = 1000

type IssFactory struct {
	cache *lru.Cache
}

func NewIssCacherFactory() (factory *IssFactory) {
	factory = &IssFactory{
		cache: lru.New(issCacheEntries, issTimeToLive),
	}
	return
}

type IssClient struct {
	cacherCert       *tls.Certificate
	cacherHost       string
	cacherPort       int
	cache            *lru.Cache
	connected        bool
	transportFactory thrift.TTransportFactory
	protocolFactory  thrift.TProtocolFactory
	transport        thrift.TTransport
	serviceClient    *IssServiceClient
}

type IssInstance struct {
	Slot               string
	Configuration      string
	NannyServiceId     string
	QloudComponent     string
	QloudInstallation  string
	ContainerFqdn      string
	GencfgGroup        string
	YasmTags           map[string]string
	HasSlotCpuLimit    bool
	HasSlotMemoryLimit bool
	FromNanny          bool
	FromQloud          bool
}

func (f *IssFactory) NewClient(cert *tls.Certificate, host string, port int) (client *IssClient, err error) {
	client = new(IssClient)
	client.cacherCert = cert
	client.cacherHost = host
	client.cacherPort = port
	client.cache = f.cache
	client.connected = false
	return
}

func (client *IssClient) connect() (err error) {
	if client.connected {
		return
	}

	client.transportFactory = thrift.NewTBufferedTransportFactory(8192)
	client.transportFactory = thrift.NewTFramedTransportFactory(client.transportFactory)
	client.protocolFactory = thrift.NewTBinaryProtocolFactoryDefault()

	addr, err := net.ResolveTCPAddr("tcp6", fmt.Sprintf(`%s:%d`, client.cacherHost, client.cacherPort))
	if err != nil {
		return
	}

	timeout := 15 * time.Second
	if client.cacherCert != nil {
		cfg := tls.Config{Certificates: []tls.Certificate{*client.cacherCert}, InsecureSkipVerify: true, ServerName: client.cacherHost}
		client.transport = thrift.NewTSSLSocketFromAddrTimeout(addr, &cfg, timeout)
	} else {
		client.transport = thrift.NewTSocketFromAddrTimeout(addr, timeout)
	}

	client.transport, err = client.transportFactory.GetTransport(client.transport)
	if err != nil {
		return
	}

	err = client.transport.Open()
	if err != nil {
		return
	}

	inputProtocol := client.protocolFactory.GetProtocol(client.transport)
	outputProtocol := client.protocolFactory.GetProtocol(client.transport)
	client.serviceClient = NewIssServiceClient(thrift.NewTStandardClient(inputProtocol, outputProtocol))
	client.connected = true
	return
}

func (c *IssClient) GetHostInstances(ctx context.Context, host string) (instances []IssInstance, err error) {
	now := time.Now()
	cacheKey := fmt.Sprintf(`%s\x00%s`, host, c.cacherHost)
	if value, ok := c.cache.Get(cacheKey, now); ok {
		instances = value.([]IssInstance)
		return
	}

	err = c.connect()
	if err != nil {
		return
	}

	response, err := c.serviceClient.GetHostConfiguration(ctx, host, "")
	if err != nil {
		return
	}

	for _, instance := range response.Instances {
		if instance.TargetState != "ACTIVE" {
			continue
		}
		slot := string(instance.ID.Slot)
		configuration := string(instance.ID.Configuration)
		var nannyServiceId string
		var qloudComponent string
		var qloudInstallation string
		var containerFqdn string
		var gencfgGroup string
		var yasmTags map[string]string
		var fromNanny bool
		var fromQloud bool
		hasSlotCpuLimit := false
		hasSlotMemoryLimit := false
		for key, value := range instance.Properties {
			if key == "properties/NANNY_SERVICE_ID" {
				nannyServiceId = value
				fromNanny = true
			} else if key == "properties/qloudDiscoveryComponent" {
				qloudComponent = value
			} else if key == "properties/qloudInstallation" {
				qloudInstallation = value
				fromQloud = true
			} else if key == "properties/HOSTNAME" {
				containerFqdn = value
			} else if key == "dynamicProperties/GENCFG_GROUP" {
				gencfgGroup = value
			} else if key == "properties/tags" {
				yasmTags = util.ExtractYasmTags(value)
			} else if key == "properties/all-tags" {
				yasmTags = util.ExtractYasmTags(value)
			} else if key == "container/constraints/slot.cpu_limit" {
				hasSlotCpuLimit = true
			} else if key == "container/constraints/slot.memory_limit" {
				hasSlotMemoryLimit = true
			}
		}
		if nannyServiceId != "" {
			yasmTags["nanny"] = nannyServiceId
		}
		if gencfgGroup != "" {
			yasmTags["gencfg"] = gencfgGroup
		}
		instances = append(instances, IssInstance{
			Slot:               slot,
			Configuration:      configuration,
			NannyServiceId:     nannyServiceId,
			QloudComponent:     qloudComponent,
			QloudInstallation:  qloudInstallation,
			ContainerFqdn:      containerFqdn,
			GencfgGroup:        gencfgGroup,
			YasmTags:           yasmTags,
			HasSlotCpuLimit:    hasSlotCpuLimit,
			HasSlotMemoryLimit: hasSlotMemoryLimit,
			FromNanny:          fromNanny,
			FromQloud:          fromQloud,
		})
	}
	c.cache.Add(cacheKey, instances, now)

	return
}

func (c *IssClient) Close() error {
	if c.connected {
		c.connected = false
		return c.transport.Close()
	}
	return nil
}
