package nanny

import (
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"net/http"
	"time"

	"golang.org/x/net/context"

	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/lru"
)

var nannyTimeToLive, _ = time.ParseDuration("30m")

const nannyCacheEntries = 1000

type NannyPersons struct {
	Logins []string `json:"logins"`
	Groups []string `json:"groups"`
}

type NannyServiceAuthAttrs struct {
	Observers    NannyPersons `json:"observers"`
	Owners       NannyPersons `json:"owners"`
	ConfManagers NannyPersons `json:"conf_managers"`
	OpsManagers  NannyPersons `json:"ops_managers"`
}

type NannyServiceAuthAttrsReply struct {
	Content NannyServiceAuthAttrs `json:"content"`
	Error   string                `json:"error"`
	Message string                `json:"msg"`
}

type NannyServiceInfoAttrs struct {
	AbcGroup                 *int64  `json:"abc_group"`
	MaintenanceNotifications *string `json:"maintenance_notifications"`
}

type NannyServiceInfoAttrsReply struct {
	Content NannyServiceInfoAttrs `json:"content"`
	Error   string                `json:"error"`
	Message string                `json:"msg"`
}

type NannyAttrsReply interface {
	GetError() string
	GetMessage() string
}

func (r *NannyServiceAuthAttrsReply) GetError() string {
	return r.Error
}

func (r *NannyServiceAuthAttrsReply) GetMessage() string {
	return r.Message
}

func (r *NannyServiceInfoAttrsReply) GetError() string {
	return r.Error
}

func (r *NannyServiceInfoAttrsReply) GetMessage() string {
	return r.Message
}

type NannyClient struct {
	client    *http.Client
	token     string
	authCache *lru.Cache
	infoCache *lru.Cache
}

func NewNannyClient(token string) (client *NannyClient, err error) {
	client = new(NannyClient)
	client.client = &http.Client{Timeout: time.Duration(15 * time.Second)}
	client.token = token
	client.authCache = lru.New(nannyCacheEntries, nannyTimeToLive)
	client.infoCache = lru.New(nannyCacheEntries, nannyTimeToLive)
	return
}

func (c *NannyClient) makeRequest(ctx context.Context, path string) (responseBody []byte, err error) {
	req, err := http.NewRequest("GET", fmt.Sprintf(`https://nanny.yandex-team.ru%s`, path), nil)
	if err != nil {
		return
	}

	req.Header.Set("Accept", "application/json")
	req.Header.Set("Authorization", fmt.Sprintf(`OAuth %s`, c.token))
	req = req.WithContext(ctx)
	resp, err := c.client.Do(req)
	if err != nil {
		return
	}
	if resp.StatusCode != 200 {
		err = fmt.Errorf("request to nanny failed with code %d", resp.StatusCode)
		return
	}

	defer func() { _ = resp.Body.Close() }()
	responseBody, err = ioutil.ReadAll(resp.Body)
	return
}

func (c *NannyClient) makeAndDecodeRequest(ctx context.Context, path string, result interface{}) error {
	responseBody, err := c.makeRequest(ctx, path)
	if err != nil {
		return err
	}

	err = json.Unmarshal(responseBody, result)
	if err != nil {
		return err
	}

	attrs := result.(NannyAttrsReply)
	if attrs.GetError() == "Not found" {
		return nil
	}

	if attrs.GetError() != "" {
		return errors.New(attrs.GetMessage())
	}

	return nil
}

func (c *NannyClient) GetServiceAuthAttrs(ctx context.Context, serviceID string) (authAttrs *NannyServiceAuthAttrs, err error) {
	now := time.Now()

	if value, ok := c.authCache.Get(serviceID, now); ok {
		authAttrs = value.(*NannyServiceAuthAttrs)
		return
	}

	var result NannyServiceAuthAttrsReply
	err = c.makeAndDecodeRequest(ctx, fmt.Sprintf("/v2/services/%s/auth_attrs/", serviceID), &result)
	if err != nil {
		return
	}

	authAttrs = &result.Content
	c.authCache.Add(serviceID, authAttrs, now)
	return
}

func (c *NannyClient) GetServiceInfoAttrs(ctx context.Context, serviceID string) (infoAttrs *NannyServiceInfoAttrs, err error) {
	now := time.Now()

	if value, ok := c.infoCache.Get(serviceID, now); ok {
		infoAttrs = value.(*NannyServiceInfoAttrs)
		return
	}

	var result NannyServiceInfoAttrsReply
	err = c.makeAndDecodeRequest(ctx, fmt.Sprintf("/v2/services/%s/info_attrs/", serviceID), &result)
	if err != nil {
		return
	}

	infoAttrs = &result.Content
	c.infoCache.Add(serviceID, infoAttrs, now)
	return
}
