package netmon

import (
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"net/http"
	"strings"
	"time"

	"golang.org/x/net/context"
)

type InvalidHostError struct {
}

func (e *InvalidHostError) Error() string {
	return "Host not exists"
}

type NetmonHostInfo struct {
	Name           string `json:"name"`
	DatacenterName string `json:"dc"`
	QueueName      string `json:"queue"`
	SwitchName     string `json:"switch"`
}

type NetmonHostsRequest struct {
	Hosts []string `json:"hosts"`
}

type NetmonHostsReply struct {
	Hosts map[string]NetmonHostInfo `json:"hosts"`
}

type NetmonExpressionReply struct {
	Hosts   []string `json:"hosts"`
	Partial bool     `json:"partial"`
	Failed  bool     `json:"failed"`
}

type NetmonClient struct {
	Client *http.Client
}

func NewNetmonClient() (client *NetmonClient, err error) {
	client = new(NetmonClient)
	client.Client = &http.Client{Timeout: time.Duration(15 * time.Second)}
	return
}

func (c *NetmonClient) makeRequest(ctx context.Context, path string, requestBody string) (responseBody []byte, err error) {
	url := fmt.Sprintf(`https://netmon-resolver.in.yandex-team.ru%s`, path)
	req, err := http.NewRequest("POST", url, strings.NewReader(requestBody))
	if err != nil {
		return
	}

	req.Header.Set("Content-Type", "application/json")
	req = req.WithContext(ctx)
	resp, err := c.Client.Do(req)
	if err != nil {
		return
	}

	defer resp.Body.Close()
	responseBody, err = ioutil.ReadAll(resp.Body)
	if resp.StatusCode == 400 {
		err = &InvalidHostError{}
	} else if resp.StatusCode != 200 {
		err = fmt.Errorf("request to netmon failed with code %d", resp.StatusCode)
	}
	return
}

func (c *NetmonClient) GetHostsInExpression(ctx context.Context, expression string) (hosts []string, err error) {
	requestBody := fmt.Sprintf(`{"expression": "%s"}`, expression)
	responseBody, err := c.makeRequest(ctx, "/api/expression/v1/expand", requestBody)
	if err != nil {
		return
	}

	var result NetmonExpressionReply
	err = json.Unmarshal(responseBody, &result)
	if err != nil {
		return
	}

	if result.Failed || result.Partial {
		err = errors.New("partial or failed response from netmon")
		return
	}

	hosts = result.Hosts
	return
}

func (c *NetmonClient) GetHostsInSwitch(ctx context.Context, switchName string) (hosts []string, err error) {
	expr := fmt.Sprintf(`switch=%s`, switchName)
	hosts, err = c.GetHostsInExpression(ctx, expr)
	return
}

func (c *NetmonClient) GetHostsInfo(ctx context.Context, hosts []string) (hostsInfo []NetmonHostInfo, err error) {
	requestBody, err := json.Marshal(&NetmonHostsRequest{Hosts: hosts})
	if err != nil {
		return
	}

	responseBody, err := c.makeRequest(ctx, "/api/v1/hosts", string(requestBody))
	if err != nil {
		return
	}

	var result NetmonHostsReply
	err = json.Unmarshal(responseBody, &result)
	if err != nil {
		return
	}

	hostsInfo = make([]NetmonHostInfo, 0, len(result.Hosts))
	for _, info := range result.Hosts {
		hostsInfo = append(hostsInfo, info)
	}
	return
}

func (c *NetmonClient) GetOneHostInfo(ctx context.Context, host string) (result *NetmonHostInfo, err error) {
	hostsInfo, err := c.GetHostsInfo(ctx, []string{host})
	if err != nil {
		return
	}

	for _, info := range hostsInfo {
		if info.Name == host {
			result = &info
			return
		}
	}

	err = fmt.Errorf("host \"%s\" not exists", host)
	return
}
