package qloud

import (
	"context"
	"fmt"
	"io/ioutil"
	"net/http"
	"strings"
	"sync"
	"time"

	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/lru"
)

const qloudCacheEntries = 1000

var qloudTimeToLive, _ = time.ParseDuration("30m")
var qloudAccessToken string

var clientsMtx = sync.Mutex{}
var clients = make(map[string]*Client)

type Client struct {
	client  *http.Client
	segment string
	token   string
	cache   *lru.Cache
}

func newClient(segment string) *Client {
	return &Client{
		client:  &http.Client{Timeout: time.Duration(15 * time.Second)},
		segment: segment,
		token:   getAccessToken(segment),
		cache:   lru.New(qloudCacheEntries, qloudTimeToLive),
	}
}

func GetClient(environment *Environment) *Client {
	clientsMtx.Lock()
	defer clientsMtx.Unlock()
	if _, ok := clients[environment.Segment]; ok {
		return clients[environment.Segment]
	}

	c := newClient(environment.Segment)
	clients[environment.Segment] = c
	return c
}

func ConfigureClients(token string) {
	qloudAccessToken = token
}

func (c *Client) getRequest(ctx context.Context, path string) ([]byte, error) {
	req, err := http.NewRequest("GET", fmt.Sprintf("https://%s.yandex-team.ru%s", c.segment, path), nil)
	if err != nil {
		return nil, err
	}

	req.Header.Set("Accept", "application/json")
	req.Header.Set("Authorization", fmt.Sprintf(`OAuth %s`, c.token))
	req = req.WithContext(ctx)

	resp, err := c.client.Do(req)
	if err != nil {
		return nil, err
	}
	if resp.StatusCode != 200 {
		return nil, fmt.Errorf("request to qloud failed with code %d", resp.StatusCode)
	}

	responseBody, err := ioutil.ReadAll(resp.Body)

	if err != nil {
		return nil, err
	}

	if err := resp.Body.Close(); err != nil {
		return nil, err
	}

	return responseBody, nil
}

func (c *Client) getACL(ctx context.Context, environment *Environment) (*ACL, error) {
	now := time.Now()
	if value, ok := c.cache.Get(environment.dnsSD, now); ok {
		return value.(*ACL), nil
	}

	resp, err := c.getRequest(ctx, fmt.Sprintf("/api/v1/access/%s.%s.%s", environment.Project, environment.Application, environment.Environment))
	if err != nil {
		return nil, err
	}

	acl, err := decodeACL(resp)

	if err != nil {
		return nil, err
	}

	c.cache.Add(environment.dnsSD, acl, now)

	return acl, nil
}

func (c *Client) CollectACL(ctx context.Context, environment *Environment) (*ACL, error) {
	result := ACL{}
	fullACL, err := c.getACL(ctx, environment)
	if err != nil {
		return nil, err
	}
	result.ObjectID = fullACL.ObjectID
	result.ObjectName = fullACL.ObjectName
	result.ACEs = make([]ACE, 0)
	for _, ace := range fullACL.ACEs {
		if !strings.EqualFold(ace.GrantedObjectLevel, "global") {
			result.ACEs = append(result.ACEs, ace)
		}
	}
	return &result, nil
}

// stub for case when qloud installations will require different tokens
func getAccessToken(segment string) string {
	return qloudAccessToken
}
