package qyp

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"time"

	"golang.org/x/net/context"

	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/lru"
)

var qypTimeToLive, _ = time.ParseDuration("30m")

const qypCacheEntries = 1000

var qypClusters = map[string]string{
	"YP_SAS_TEST": "vmproxy.test-swat.yandex-team.ru",
	"YP_MAN_PRE":  "vmproxy.pre-swat.yandex-team.ru",
	"YP_IVA":      "vmproxy.iva-swat.yandex-team.ru",
	"YP_MAN":      "vmproxy.man-swat.yandex-team.ru",
	"YP_MYT":      "vmproxy.myt-swat.yandex-team.ru",
	"YP_SAS":      "vmproxy.sas-swat.yandex-team.ru",
	"YP_VLA":      "vmproxy.vla-swat.yandex-team.ru",
}

var qypClusterNames = map[string]string{
	"YP_SAS_TEST": "test",
	"YP_MAN_PRE":  "pre",
	"YP_IVA":      "iva",
	"YP_MAN":      "man",
	"YP_MYT":      "myt",
	"YP_SAS":      "sas",
	"YP_VLA":      "vla",
}

type QypOwners struct {
	Logins []string `json:"logins"`
	Groups []string `json:"groupIds"`
}

type QypAuth struct {
	Owners QypOwners `json:"owners"`
}

type QypMeta struct {
	Auth QypAuth `json:"auth"`
	ID   string  `json:"id"`
}

type QypVM struct {
	Meta QypMeta `json:"meta"`
}

type QypListYpVMReply struct {
	VMs []QypVM `json:"vms"`
}

type QypClient struct {
	client *http.Client
	token  string
	cache  *lru.Cache
}

func NewQypClient(token string) (client *QypClient, err error) {
	client = new(QypClient)
	client.client = &http.Client{Timeout: time.Duration(15 * time.Second)}
	client.token = token
	client.cache = lru.New(qypCacheEntries, qypTimeToLive)
	return
}

func (c *QypClient) makeRequest(ctx context.Context, cluster string, path string) (responseBody []byte, err error) {
	backend, ok := qypClusters[cluster]
	if !ok {
		err = fmt.Errorf("no QYP backend exists for %s", cluster)
		return
	}

	req, err := http.NewRequest("GET", fmt.Sprintf(`https://%s%s`, backend, path), nil)
	if err != nil {
		return
	}

	req.Header.Set("Accept", "application/json")
	req.Header.Set("Authorization", fmt.Sprintf(`OAuth %s`, c.token))
	req = req.WithContext(ctx)
	resp, err := c.client.Do(req)
	if err != nil {
		return
	}

	if resp.StatusCode != 200 {
		err = fmt.Errorf("%s replied with code %d", backend, resp.StatusCode)
		return
	}

	defer func() { _ = resp.Body.Close() }()
	responseBody, err = ioutil.ReadAll(resp.Body)
	return
}

func (c *QypClient) makeAndDecodeRequest(ctx context.Context, cluster string, path string, result interface{}) error {
	responseBody, err := c.makeRequest(ctx, cluster, path)
	if err != nil {
		return err
	}

	err = json.Unmarshal(responseBody, result)
	if err != nil {
		return err
	}

	return nil
}

func (c *QypClient) GetOwners(ctx context.Context, cluster string, vmName string) (owners *QypOwners, err error) {
	now := time.Now()

	cacheKey := fmt.Sprintf(`%s\x00%s`, cluster, vmName)
	if value, ok := c.cache.Get(cacheKey, now); ok {
		owners = value.(*QypOwners)
		return
	}

	var result QypListYpVMReply
	err = c.makeAndDecodeRequest(ctx, cluster, fmt.Sprintf("/api/ListYpVm/?query.name=%s", vmName), &result)
	if err != nil {
		return
	}

	if len(result.VMs) == 0 {
		return
	}

	for _, vm := range result.VMs {
		if vm.Meta.ID == vmName {
			owners = &vm.Meta.Auth.Owners
			c.cache.Add(cacheKey, owners, now)
			return
		}
	}

	err = fmt.Errorf("vm %s not found", vmName)
	return
}

func (c *QypClient) GetClusterName(cluster string) string {
	name, ok := qypClusterNames[cluster]
	if !ok {
		return "none"
	}
	return name
}
