package staff

import (
	"encoding/gob"
	"errors"
	"fmt"
	"os"
	"time"

	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/abc"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/freshcache"
)

type CacheDriver struct {
	client    *StaffClient
	abcClient *abc.AbcClient
}

func (c *CacheDriver) Update() error {
	c.client.cacheMutex.Lock()
	defer c.client.cacheMutex.Unlock()

	cache := newCache()

	err := c.abcClient.FetchAllMembers(func(member abc.AbcMember) {
		cache.addDirectMemberToService(member.Person.Login, member.Service.ID)
	})
	if err != nil {
		return err
	}

	err = c.client.fetchAllPeople(func(person StaffPerson) {
		cache.addPerson(person)
	})
	if err != nil {
		return err
	}

	cache.cleanup()
	cache.Timestamp = time.Now()
	c.client.cache = cache

	return nil
}

func (c *CacheDriver) LoadState(path string) error {
	c.client.cacheMutex.Lock()
	defer c.client.cacheMutex.Unlock()

	cacheFile, err := os.Open(path)
	if err != nil {
		return err
	}
	defer func() { _ = cacheFile.Close() }()

	cache := new(Cache)
	dec := gob.NewDecoder(cacheFile)
	err = dec.Decode(cache)
	if err != nil {
		return err
	}
	if !cache.isCompatible() {
		return errors.New("staff cache isn't compatible")
	}

	c.client.cache = cache

	return nil
}

func (c *CacheDriver) DumpState(path string) error {
	c.client.cacheMutex.Lock()
	defer c.client.cacheMutex.Unlock()

	if c.client.cache == nil {
		return fmt.Errorf("staff cache not initialized")
	}

	temporaryPath := fmt.Sprintf("%s.tmp", path)
	temporaryFile, err := os.Create(temporaryPath)
	if err != nil {
		return err
	}

	enc := gob.NewEncoder(temporaryFile)
	err = enc.Encode(*c.client.cache)
	if err != nil {
		return err
	}

	err = temporaryFile.Close()
	if err != nil {
		return err
	}

	return os.Rename(temporaryPath, path)
}

func (c *CacheDriver) GetAge() time.Duration {
	if c.client.cache == nil {
		return time.Duration(0)
	}
	return time.Since(c.client.cache.Timestamp)
}

func (c *CacheDriver) GetOptions() freshcache.CacheDriverOptions {
	return freshcache.CacheDriverOptions{
		Name:        "staff",
		RefreshTime: 60 * time.Minute,
	}
}

func (c *CacheDriver) GetCache() interface{} {
	return c.client
}

func (c *CacheDriver) Stop() {
	c.client.cancel()
}

func NewCacheDriver(staffClient *StaffClient, abcClient *abc.AbcClient) *CacheDriver {
	return &CacheDriver{client: staffClient, abcClient: abcClient}
}
