package staff

import (
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"net/http"
	"strconv"
	"sync"
	"time"

	"golang.org/x/net/context"

	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/log"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/util"
	aLog "a.yandex-team.ru/library/go/core/log"
)

const staffLimit = 1000

type StaffPersonOfficial struct {
	IsDismissed  bool `json:"is_dismissed"`
	IsRobot      bool `json:"is_robot"`
	IsHomeworker bool `json:"is_homeworker"`
}

type StaffPerson struct {
	IsDeleted bool                `json:"is_deleted"`
	Official  StaffPersonOfficial `json:"official"`
	UID       string              `json:"uid"`
	Login     string              `json:"login"`
	ID        int64               `json:"id"`
	Groups    []StaffPersonGroup  `json:"groups"`
}

type StaffPersonGroup struct {
	Group StaffPersonGroupDetail `json:"group"`
}

type StaffPersonGroupDetail struct {
	ID        int64                    `json:"id"`
	Type      string                   `json:"type"`
	RoleScope *string                  `json:"role_scope"`
	URL       string                   `json:"url"`
	Parent    *StaffPersonGroupDetail  `json:"parent"`
	Service   *StaffPersonGroupService `json:"service"`
}

type StaffPersonGroupService struct {
	ID int64 `json:"id"`
}

type StaffPersonReply struct {
	Result []StaffPerson `json:"result"`
	Pages  int64         `json:"pages"`
	Total  int64         `json:"total"`
	Limit  int64         `json:"limit"`
	Page   int64         `json:"page"`
}

type StaffClient struct {
	client     *http.Client
	token      string
	cache      *Cache
	cacheMutex sync.Mutex
	ctx        context.Context
	cancel     context.CancelFunc
}

func NewStaffClient(token string) (client *StaffClient, err error) {
	ctx := context.Background()
	ctx, cancel := context.WithCancel(ctx)
	client = new(StaffClient)
	client.client = &http.Client{Timeout: time.Duration(60 * time.Second)}
	client.token = token
	client.cache = nil
	client.ctx = ctx
	client.cancel = cancel
	return
}

func (c *StaffClient) makeRequest(path string) (resp *http.Response, err error) {
	req, err := http.NewRequest("GET", fmt.Sprintf(`https://staff-api.yandex-team.ru%s`, path), nil)
	if err != nil {
		return
	}

	req.Header.Set("Accept", "application/json")
	req.Header.Set("Authorization", fmt.Sprintf(`OAuth %s`, c.token))
	resp, err = c.client.Do(req)
	return
}

func (c *StaffClient) fetchPeople(fromID int64, cb func(StaffPerson)) error {
	url := fmt.Sprintf("/v3/persons?&_fields="+
		"id,uid,login,is_deleted,"+
		"official.is_dismissed,official.is_robot,official.is_homeworker,"+
		"groups.group.id,groups.group.type,groups.group.url,groups.group.role_scope,"+
		"groups.group.parent.id,groups.group.parent.service.id,groups.group.service.id"+
		"&_sort=id&_limit=%d&_query=id>%d", staffLimit, fromID)
	resp, err := c.makeRequest(url)
	if err != nil {
		return err
	}
	if resp.StatusCode != 200 {
		return fmt.Errorf("request to staff failed with code %d", resp.StatusCode)
	}

	defer func() { _ = resp.Body.Close() }()

	dec := json.NewDecoder(resp.Body)
	for {
		t, err := dec.Token()
		if err == io.EOF {
			break
		}
		if err != nil {
			return err
		}
		switch v := t.(type) {
		case string:
			if v == "result" {
				goto ResultFound
			}
		default:
		}
	}

ResultFound:
	_, err = dec.Token()
	if err != nil {
		return err
	}

	for dec.More() {
		person := StaffPerson{}
		err = dec.Decode(&person)
		if err != nil {
			return err
		}

		cb(person)
	}

	for {
		_, err := dec.Token()
		if err == io.EOF {
			break
		}
		if err != nil {
			return err
		}
	}

	return nil
}

func (c *StaffClient) fetchPeopleWithRetry(fromID int64, cb func(StaffPerson)) error {
	// TODO: request only new records
	var err error

	for retry := 1; retry <= 3; retry++ {
		err = c.fetchPeople(fromID, cb)
		if err == nil {
			break
		}

		sleepDuration := util.RetryBackoff(retry, time.Second, time.Minute)
		select {
		case <-c.ctx.Done():
			return errors.New("staff cache update cancelled")
		case <-time.After(sleepDuration):
			continue
		}
	}

	return err
}

func (c *StaffClient) fetchAllPeople(cb func(StaffPerson)) error {
	var lastID, prevID int64
	lastID = -1
	prevID = -1

	wrappedCb := func(person StaffPerson) {
		cb(person)
		lastID = person.ID
	}

	err := c.fetchPeople(lastID, wrappedCb)
	for err == nil && lastID != prevID {
		prevID = lastID
		log.Logger.Debug("fetching persons", aLog.Int64("lastId", lastID))
		err = c.fetchPeopleWithRetry(lastID, wrappedCb)
	}

	return err
}

func (c *StaffClient) GetPersonsInGroup(groupID string) ([]StaffPerson, error) {
	groupID64, err := strconv.ParseInt(groupID, 10, 64)
	if err != nil {
		return nil, err
	}

	cache := c.cache
	if cache == nil || !cache.isActual() {
		return nil, fmt.Errorf("staff cache not initialized or outdated")
	}

	persons, err := cache.findPeopleForGroup(groupID64)
	if err != nil {
		return nil, err
	}

	return persons, nil
}

func (c *StaffClient) GetPersonsInService(abcGroup int64) ([]StaffPerson, error) {
	cache := c.cache
	if cache == nil || !cache.isActual() {
		return nil, fmt.Errorf("staff cache not initialized or outdated")
	}

	persons, err := cache.findAdmins(abcGroup)
	if err != nil {
		return nil, err
	}

	return persons, nil
}

func (c *StaffClient) GetPerson(login string) (*StaffPerson, error) {
	cache := c.cache
	if cache == nil || !cache.isActual() {
		return nil, fmt.Errorf("staff cache not initialized or outdated")
	}

	if person, ok := cache.LoginIndex[login]; ok {
		return person, nil
	}

	return nil, fmt.Errorf("person %s not found", login)
}
