package staff

import (
	"fmt"
	"time"

	aLog "a.yandex-team.ru/library/go/core/log"

	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/log"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/util"
)

var staffTimeToLive, _ = time.ParseDuration("3h")

const staffCacheVersion = 5
const adminRole string = "administration"
const devopsRole string = "devops"

type Cache struct {
	StringStorage *util.StringIntern
	LoginIndex    map[string]*StaffPerson
	GroupIndex    map[int64]CacheGroup
	ServiceIndex  map[int64]CacheService
	Timestamp     time.Time
	Version       int64
}

type CacheGroup struct {
	People    []*StaffPerson
	ServiceID int64
	RoleScope *string
}

type ServiceMember struct {
	Person     *StaffPerson
	RoleScopes []*string
}

type RoleMember struct {
	People []*StaffPerson
}

type CacheService struct {
	DirectMembers  map[string]bool
	ServiceMembers map[string]ServiceMember
	RoleMembers    map[string]RoleMember
}

func (c *CacheService) getServiceMember(person *StaffPerson) (member ServiceMember, exists bool) {
	exists = false
	if existedMember, ok := c.ServiceMembers[person.Login]; ok {
		member = existedMember
		exists = true
		return
	}
	if _, ok := c.DirectMembers[person.Login]; ok {
		member.Person = person
		member.RoleScopes = make([]*string, 0)
		exists = true
		return
	}
	return
}

func (c *CacheService) appendPersonToRole(person *StaffPerson, roleScope *string) {
	var roleMember RoleMember
	if val, ok := c.RoleMembers[*roleScope]; ok {
		roleMember = val
	} else {
		roleMember = RoleMember{
			People: make([]*StaffPerson, 0),
		}
	}
	roleMember.People = append(roleMember.People, person)
	c.RoleMembers[*roleScope] = roleMember
}

func newCache() *Cache {
	cache := &Cache{
		StringStorage: util.NewStringIntern(),
		LoginIndex:    make(map[string]*StaffPerson),
		GroupIndex:    make(map[int64]CacheGroup),
		ServiceIndex:  make(map[int64]CacheService),
		Version:       staffCacheVersion,
	}
	return cache
}

func (c *Cache) isActual() bool {
	return time.Now().Before(c.Timestamp.Add(staffTimeToLive))
}

func (c *Cache) isCompatible() bool {
	return c.Version == staffCacheVersion
}

func (c *Cache) addPerson(person StaffPerson) {
	ptr := &person
	c.LoginIndex[person.Login] = ptr
	c.parseGroups(ptr)
	ptr.Groups = nil
}

func (c *Cache) parseGroups(person *StaffPerson) {
	for _, group := range person.Groups {
		switch group.Group.Type {
		case "servicerole":
			c.parseServiceRoleGroup(person, &(group.Group))
		case "department":
			c.parseDepartment(person, &(group.Group))
		case "service":
			c.parseService(person, &(group.Group))
		}
	}
}

func (c *Cache) parseServiceRoleGroup(person *StaffPerson, group *StaffPersonGroupDetail) {
	roleScope := c.StringStorage.Intern(*group.RoleScope)
	if group.Parent != nil && group.Parent.Service != nil {
		c.GroupIndex[group.ID] = CacheGroup{
			People:    make([]*StaffPerson, 0),
			ServiceID: group.Parent.Service.ID,
			RoleScope: roleScope,
		}
		c.addPersonToService(person, group.Parent, roleScope)
	} else {
		log.Logger.Error("servicerole group has no parent or abc service", aLog.Int64("id", group.ID), aLog.String("url", group.URL))
	}
}

func (c *Cache) parseDepartment(person *StaffPerson, group *StaffPersonGroupDetail) {
	currentGroup := c.getGroup(group)
	currentGroup.People = append(c.GroupIndex[group.ID].People, person)
	c.GroupIndex[group.ID] = currentGroup
}

func (c *Cache) parseService(person *StaffPerson, group *StaffPersonGroupDetail) {
	if _, ok := c.GroupIndex[group.ID]; !ok {
		c.GroupIndex[group.ID] = CacheGroup{
			People:    make([]*StaffPerson, 0),
			ServiceID: group.Service.ID,
			RoleScope: nil,
		}
	} else {
		currentGroup := c.GroupIndex[group.ID]
		if currentGroup.ServiceID != group.Service.ID || len(currentGroup.People) > 0 {
			log.Logger.Error("service group id changed or has people in it", aLog.Int64("current", currentGroup.ServiceID), aLog.Int64("new", group.Service.ID))
		}
	}
}

func (c *Cache) getGroup(group *StaffPersonGroupDetail) CacheGroup {
	if _, ok := c.GroupIndex[group.ID]; !ok {
		currentGroup := CacheGroup{
			People:    make([]*StaffPerson, 0),
			ServiceID: 0,
			RoleScope: nil,
		}
		c.GroupIndex[group.ID] = currentGroup
		return currentGroup
	} else {
		return c.GroupIndex[group.ID]
	}
}

func (c *Cache) getService(serviceID int64) CacheService {
	if _, ok := c.ServiceIndex[serviceID]; !ok {
		currentService := CacheService{
			DirectMembers:  make(map[string]bool),
			ServiceMembers: make(map[string]ServiceMember),
			RoleMembers:    make(map[string]RoleMember),
		}
		c.ServiceIndex[serviceID] = currentService
		return currentService
	} else {
		return c.ServiceIndex[serviceID]
	}
}

func (c *Cache) addPersonToService(person *StaffPerson, group *StaffPersonGroupDetail, roleScope *string) {
	currentService := c.getService(group.Service.ID)
	if member, ok := currentService.getServiceMember(person); ok {
		member.RoleScopes = append(member.RoleScopes, roleScope)
		currentService.ServiceMembers[person.Login] = member
		currentService.appendPersonToRole(person, roleScope)
		c.ServiceIndex[group.Service.ID] = currentService
	}
}

func (c *Cache) addDirectMemberToService(login string, serviceID int64) {
	currentService := c.getService(serviceID)
	currentService.DirectMembers[login] = true
	c.ServiceIndex[serviceID] = currentService
}

func (c *Cache) cleanup() {
	for key, value := range c.ServiceIndex {
		value.DirectMembers = nil
		c.ServiceIndex[key] = value
	}
}

func (c *Cache) findPeopleForGroup(id int64) ([]StaffPerson, error) {
	if _, ok := c.GroupIndex[id]; !ok {
		return nil, fmt.Errorf("group %d not found in cache", id)
	}

	group := c.GroupIndex[id]
	if group.ServiceID > 0 && group.RoleScope != nil {
		return c.findPeopleForRole(group.ServiceID, *group.RoleScope)
	} else if group.ServiceID > 0 {
		return c.findAdmins(group.ServiceID)
	} else {
		result := make([]StaffPerson, 0)
		for _, person := range group.People {
			result = append(result, *person)
		}
		return result, nil
	}
}

func (c *Cache) findService(id int64) (*CacheService, error) {
	if service, ok := c.ServiceIndex[id]; ok {
		return &service, nil
	}
	return nil, fmt.Errorf("abc service %d not found in cache", id)
}

func (c *Cache) findPeopleForRole(id int64, roleScope string) ([]StaffPerson, error) {
	service, err := c.findService(id)
	if err != nil {
		return nil, err
	}

	result := make([]StaffPerson, 0)
	if roleMember, ok := service.RoleMembers[roleScope]; ok {
		for _, person := range roleMember.People {
			result = append(result, *person)
		}
	}
	return result, nil
}

func (c *Cache) findAdmins(id int64) ([]StaffPerson, error) {
	result := make([]StaffPerson, 0)

	admins, err := c.findPeopleForRole(id, adminRole)
	if err != nil {
		return nil, err
	}
	result = append(result, admins...)

	devops, err := c.findPeopleForRole(id, devopsRole)
	if err != nil {
		return nil, err
	}
	result = append(result, devops...)

	return result, nil
}
