package staff

import (
	"sync"
)

type StaffState struct {
	mutex          sync.Mutex
	seenGroups     map[string]bool
	failedGroups   map[string]string
	seenLogins     map[string]bool
	failedLogins   map[string]string
	personsInGroup map[string][]StaffPerson
	personByLogin  map[string]*StaffPerson
}

func NewStaffState() (state *StaffState) {
	state = &StaffState{
		seenGroups:     make(map[string]bool),
		failedGroups:   make(map[string]string),
		seenLogins:     make(map[string]bool),
		failedLogins:   make(map[string]string),
		personsInGroup: make(map[string][]StaffPerson),
		personByLogin:  make(map[string]*StaffPerson),
	}
	return
}

func (s *StaffState) IsGroupSeen(groupID string) bool {
	s.mutex.Lock()
	defer s.mutex.Unlock()
	_, seen := s.seenGroups[groupID]
	if !seen {
		s.seenGroups[groupID] = true
	}
	return seen
}

func (s *StaffState) IsLoginSeen(login string) bool {
	s.mutex.Lock()
	defer s.mutex.Unlock()
	_, seen := s.seenLogins[login]
	if !seen {
		s.seenLogins[login] = true
	}
	return seen
}

func (s *StaffState) SetPersonsInGroup(groupID string, persons []StaffPerson) {
	s.mutex.Lock()
	defer s.mutex.Unlock()
	s.personsInGroup[groupID] = persons
}

func (s *StaffState) GetPersonsInGroup(groupID string) (persons []StaffPerson) {
	s.mutex.Lock()
	defer s.mutex.Unlock()
	persons, exists := s.personsInGroup[groupID]
	if !exists {
		persons = make([]StaffPerson, 0)
	}
	return
}

func (s *StaffState) SetPerson(login string, person *StaffPerson) {
	s.mutex.Lock()
	defer s.mutex.Unlock()
	s.personByLogin[login] = person
}

func (s *StaffState) GetPerson(login string) (person *StaffPerson) {
	s.mutex.Lock()
	defer s.mutex.Unlock()
	person = s.personByLogin[login]
	return
}

func (s *StaffState) SetGroupError(groupID string, err error) {
	s.mutex.Lock()
	defer s.mutex.Unlock()
	s.failedGroups[groupID] = err.Error()
}

func (s *StaffState) GetFailedGroups() map[string]string {
	s.mutex.Lock()
	defer s.mutex.Unlock()
	return s.failedGroups
}

func (s *StaffState) SetLoginError(login string, err error) {
	s.mutex.Lock()
	defer s.mutex.Unlock()
	s.failedLogins[login] = err.Error()
}

func (s *StaffState) GetFailedLogins() map[string]string {
	s.mutex.Lock()
	defer s.mutex.Unlock()
	return s.failedLogins
}
