package walle

import (
	"encoding/gob"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"net/http"
	"os"
	"sync"
	"time"

	"golang.org/x/net/context"

	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/netmon"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/log"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/util"
	aLog "a.yandex-team.ru/library/go/core/log"
)

var walleTimeToLive, _ = time.ParseDuration("1h")

const walleCacheVersion = 6

var walleIssAdmProjects = []string{
	"rtc-iss-master-sas",
	"rtc-iss-master-man",
	"rtc-iss-master-vla",
	"rtc-iss-master-msk",
	"rtc-iss-master-global",
	"yp-sas-test",
	"yp-man-pre",
	"yp-sas",
	"yp-vla",
	"yp-man",
	"yp-msk",
	"yp-iva",
	"yp-xdc",
	"yp-prestable",
}

type WalleHost struct {
	Name       string
	Switch     string
	Queue      string
	Datacenter string
	YpMaster   string
	IssCachers []string
	FromRtc    bool
}

type RawWalleLocation struct {
	Switch     string `json:"switch"`
	Datacenter string `json:"short_datacenter_name"`
	Queue      string `json:"short_queue_name"`
}

type RawWalleHost struct {
	InventoryNumber int64             `json:"inv"`
	Name            string            `json:"name"`
	State           string            `json:"state"`
	Project         string            `json:"project"`
	Location        *RawWalleLocation `json:"location"`
	Tags            []string          `json:"tags"`
}

func isIssAdmProject(needle string) bool {
	for _, v := range walleIssAdmProjects {
		if v == needle {
			return true
		}
	}
	return false
}

func (h *RawWalleHost) GetYpMaster() string {
	for _, tag := range h.Tags {
		switch tag {
		case "rtc.ypmaster-man_pre":
			return "YP_MAN_PRE"
		case "rtc.ypmaster-man":
			return "YP_MAN"
		case "rtc.ypmaster-sas_test":
			return "YP_SAS_TEST"
		case "rtc.ypmaster-sas":
			return "YP_SAS"
		case "rtc.ypmaster-iva":
			return "YP_IVA"
		case "rtc.ypmaster-myt":
			return "YP_MYT"
		case "rtc.ypmaster-vla":
			return "YP_VLA"
		}
	}
	if h.BelongToRtc() && h.Location != nil {
		switch h.Location.Datacenter {
		case "man":
			return "YP_MAN"
		case "sas":
			return "YP_SAS"
		case "iva":
			return "YP_IVA"
		case "myt":
			return "YP_MYT"
		case "vla":
			return "YP_VLA"
		}
	}
	return ""
}

func (h *RawWalleHost) GetIssCachers() (result []string) {
	result = make([]string, 0)
	hasIssAdm := false
	for _, tag := range h.Tags {
		switch tag {
		case "rtc.isscacher-adm":
			result = append(result, "ISS_ADM")
			hasIssAdm = true
		}
	}
	if !hasIssAdm && isIssAdmProject(h.Project) {
		result = append(result, "ISS_ADM")
	}
	if h.BelongToRtc() {
		result = append(result, "ISS_MULTI")
	}
	if h.BelongToRtc() && h.Location != nil {
		switch h.Location.Datacenter {
		case "man":
			result = append(result, "ISS_MAN")
		case "sas":
			result = append(result, "ISS_SAS")
		case "iva":
			result = append(result, "ISS_MSK")
		case "myt":
			result = append(result, "ISS_MSK")
		case "vla":
			result = append(result, "ISS_VLA")
		}
	}
	return
}

func (h *RawWalleHost) BelongToRtc() bool {
	for _, tag := range h.Tags {
		if tag == "rtc" {
			return true
		}
	}
	return false
}

type WalleHostMap struct {
	Hosts     map[string]WalleHost
	Timestamp time.Time
	Version   int64
}

func (m *WalleHostMap) IsActual() bool {
	return time.Now().Before(m.Timestamp.Add(walleTimeToLive))
}

func (m *WalleHostMap) IsCompatible() bool {
	return m.Version == walleCacheVersion
}

type WalleClient struct {
	client  *http.Client
	token   string
	ctx     context.Context
	cancel  context.CancelFunc
	hostMap *WalleHostMap
	mutex   sync.Mutex
}

func NewWalleClient(token string) (client *WalleClient) {
	ctx := context.Background()
	ctx, cancel := context.WithCancel(ctx)
	client = &WalleClient{
		client:  &http.Client{Timeout: time.Duration(300 * time.Second)},
		token:   token,
		ctx:     ctx,
		cancel:  cancel,
		hostMap: nil,
	}
	return
}

func (c *WalleClient) makeRequest(path string) (resp *http.Response, err error) {
	req, err := http.NewRequest("GET", fmt.Sprintf(`https://api.wall-e.yandex-team.ru%s`, path), nil)
	if err != nil {
		return
	}

	req.Header.Set("Accept", "application/json")
	req.Header.Set("Authorization", fmt.Sprintf(`OAuth %s`, c.token))
	req = req.WithContext(c.ctx)
	resp, err = c.client.Do(req)
	return
}

func (c *WalleClient) dumpHostMap(hostMap *WalleHostMap, filePath string) error {
	temporaryPath := fmt.Sprintf("%s.tmp", filePath)
	temporaryFile, err := os.Create(temporaryPath)
	if err != nil {
		return err
	}

	enc := gob.NewEncoder(temporaryFile)
	err = enc.Encode(*hostMap)
	if err != nil {
		return err
	}

	err = temporaryFile.Close()
	if err != nil {
		return err
	}

	return os.Rename(temporaryPath, filePath)
}

func (c *WalleClient) loadHostMap(p string) (hostMap *WalleHostMap, err error) {
	cacheFile, err := os.Open(p)
	if err != nil {
		return
	}
	defer func() { _ = cacheFile.Close() }()

	hostMap = new(WalleHostMap)
	dec := gob.NewDecoder(cacheFile)
	err = dec.Decode(hostMap)
	if err != nil {
		return
	}

	if !hostMap.IsCompatible() {
		err = errors.New("walle host map cache isn't compatible")
	}

	return
}

func (c *WalleClient) fetchPage(cursor int64, cb func(RawWalleHost)) error {
	url := fmt.Sprintf("/v1/hosts?fields=inv,name,state,location.switch,location.short_datacenter_name,location.short_queue_name,tags,project&resolve_tags=yes&limit=500&cursor=%d", cursor)
	resp, err := c.makeRequest(url)
	if err != nil {
		return err
	}
	if resp.StatusCode != 200 {
		return fmt.Errorf("request to walle failed with code %d", resp.StatusCode)
	}

	defer func() { _ = resp.Body.Close() }()

	dec := json.NewDecoder(resp.Body)
	for {
		t, err := dec.Token()
		if err == io.EOF {
			break
		}
		if err != nil {
			return err
		}
		switch v := t.(type) {
		case string:
			if v == "result" {
				goto ResultFound
			}
		default:
		}
	}

ResultFound:
	_, err = dec.Token()
	if err != nil {
		return err
	}

	for dec.More() {
		host := RawWalleHost{}
		err = dec.Decode(&host)
		if err != nil {
			return err
		}

		cb(host)
	}

	for {
		_, err := dec.Token()
		if err == io.EOF {
			break
		}
		if err != nil {
			return err
		}
	}

	return nil
}

func (c *WalleClient) fetchPageWithRetry(cursor int64, cb func(RawWalleHost)) error {
	// TODO: request only new records
	// TODO: cb may be called multiple times with same host
	var err error

	for retry := 1; retry <= 3; retry++ {
		err = c.fetchPage(cursor, cb)
		if err == nil {
			break
		}

		sleepDuration := util.RetryBackoff(retry, time.Second, time.Minute)
		select {
		case <-c.ctx.Done():
			return errors.New("walle host map update cancelled")
		case <-time.After(sleepDuration):
			continue
		}
	}

	return err
}

func (c *WalleClient) fetchHostMap() (result *WalleHostMap, err error) {
	result = &WalleHostMap{
		Hosts:     make(map[string]WalleHost),
		Timestamp: time.Now(),
		Version:   walleCacheVersion,
	}

	var prevCursor, lastCursor int64
	prevCursor = 0
	lastCursor = 0

	cb := func(host RawWalleHost) {
		if host.State != "free" && host.Location != nil {
			result.Hosts[host.Name] = WalleHost{
				Name:       host.Name,
				Switch:     host.Location.Switch,
				Datacenter: host.Location.Datacenter,
				YpMaster:   host.GetYpMaster(),
				IssCachers: host.GetIssCachers(),
				FromRtc:    host.BelongToRtc(),
			}
		}
		lastCursor = host.InventoryNumber
	}

	err = c.fetchPageWithRetry(lastCursor, cb)
	for err == nil && lastCursor != prevCursor {
		prevCursor = lastCursor
		log.Logger.Debug("fetching walle hosts", aLog.Int64("lastCursor", lastCursor))
		err = c.fetchPageWithRetry(lastCursor+1, cb)
	}
	if err != nil {
		return
	}

	return
}

func (c *WalleClient) updateHostMap() (int, error) {
	hostMap, err := c.fetchHostMap()
	if err != nil {
		return 0, err
	}

	c.mutex.Lock()
	c.hostMap = hostMap
	c.mutex.Unlock()

	return len(hostMap.Hosts), nil
}

func (c *WalleClient) getActualHostMap() (result *WalleHostMap) {
	c.mutex.Lock()
	result = c.hostMap
	c.mutex.Unlock()
	if result != nil && !result.IsActual() {
		result = nil
	}
	return
}

func (c *WalleClient) GetHostInfo(fqdn string) (result *WalleHost, err error) {
	hostMap := c.getActualHostMap()
	if hostMap == nil {
		err = errors.New("walle host map is outdated")
		return
	}

	if val, ok := hostMap.Hosts[fqdn]; ok {
		result = &val
	} else {
		result = nil
	}
	return
}

func (c *WalleClient) GetNetmonHostInfo(host string) (result *netmon.NetmonHostInfo, err error) {
	walleHostsInfo, err := c.GetHostInfo(host)
	if err != nil {
		return
	}
	if walleHostsInfo == nil {
		err = &netmon.InvalidHostError{}
		return
	}
	netmonHostInfo := netmon.NetmonHostInfo{
		Name:           walleHostsInfo.Name,
		DatacenterName: walleHostsInfo.Datacenter,
		QueueName:      walleHostsInfo.Queue,
		SwitchName:     walleHostsInfo.Switch,
	}
	return &netmonHostInfo, nil
}
