package yp

import (
	"fmt"
	"time"

	"golang.org/x/net/context"
	"google.golang.org/grpc"

	pb "a.yandex-team.ru/infra/rtc/instance_resolver/api"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/lru"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/util"
)

var ypTimeToLive, _ = time.ParseDuration("1m")

const ypCacheEntries = 5000

type YpClient struct {
	clientConn *grpc.ClientConn
	podCache   *lru.Cache
	nodeCache  *lru.Cache
}

type YpInstance struct {
	YpPodSet         string
	NannyServiceID   string
	FromQyp          bool
	FromNanny        bool
	FromDeploy       bool
	ContainerFqdn    string
	DeployEngine     string
	MaintenanceState string
	YasmTags         map[string]string
	DeployEnviron    string
	DeployUnitID     string
	DeployStageID    string
	DeployProjectID  string
}

type YpNode struct {
	NodeID           string
	Segment          string
	MaintenanceState string
}

func NewYpClient(address string) (client *YpClient, err error) {
	client = new(YpClient)
	//nolint:SA1019
	client.clientConn, err = grpc.Dial(address, grpc.WithInsecure(), grpc.WithBackoffMaxDelay(time.Second))
	client.podCache = lru.New(ypCacheEntries, ypTimeToLive)
	client.nodeCache = lru.New(ypCacheEntries, ypTimeToLive)
	return
}

func (c *YpClient) GetHostInstances(ctx context.Context, host string, origin string) (instances []YpInstance, err error) {
	now := time.Now()
	ctx, cancel := context.WithTimeout(ctx, 20*time.Second)
	defer cancel()

	cacheKey := fmt.Sprintf(`%s\x00%s`, host, origin)
	if value, ok := c.podCache.Get(cacheKey, now); ok {
		instances = value.([]YpInstance)
		return
	}

	proxyClient := pb.NewTYpProxyClient(c.clientConn)
	response, err := proxyClient.FindPodsetsByNode(ctx, &pb.TFindPodsetsByNodeRequest{NodeName: host, Origin: origin})
	if err != nil {
		return
	}

	for _, instance := range response.Instances {
		var tags map[string]string

		if instance.DeployEngine == "YP_LITE" || instance.DeployEngine == "QYP" {
			tags = util.ExtractYasmTags(instance.IssTags)
		} else if instance.DeployEngine == "RSC" || instance.DeployEngine == "MCRSC" {
			tags = instance.DeployMonitoring.Labels
		}
		if instance.DeployEngine == "YP_LITE" {
			tags["nanny"] = instance.NannyServiceId
		}

		instances = append(instances, YpInstance{
			YpPodSet:         instance.YpPodSet,
			NannyServiceID:   instance.NannyServiceId,
			FromQyp:          instance.FromQyp,
			FromDeploy:       instance.FromDeploy,
			FromNanny:        instance.FromNanny,
			ContainerFqdn:    instance.ContainerFqdn,
			DeployEngine:     instance.DeployEngine,
			MaintenanceState: instance.MaintenanceState,
			DeployEnviron:    instance.DeployEnviron,
			DeployUnitID:     instance.DeployUnitId,
			DeployStageID:    instance.DeployStageId,
			DeployProjectID:  instance.DeployProjectId,
			YasmTags:         tags,
		})
	}
	c.podCache.Add(cacheKey, instances, now)
	return
}

func (c *YpClient) GetNode(ctx context.Context, host string, origin string) (node YpNode, err error) {
	now := time.Now()
	ctx, cancel := context.WithTimeout(ctx, 20*time.Second)
	defer cancel()

	cacheKey := fmt.Sprintf(`%s\x00%s`, host, origin)
	if value, ok := c.nodeCache.Get(cacheKey, now); ok {
		node = value.(YpNode)
		return
	}

	proxyClient := pb.NewTYpProxyClient(c.clientConn)
	response, err := proxyClient.GetNode(ctx, &pb.TGetNodeRequest{NodeName: host, Origin: origin})
	if err != nil {
		return
	}

	node = YpNode{
		NodeID:           response.Node.NodeId,
		Segment:          response.Node.Segment,
		MaintenanceState: response.Node.MaintenanceState,
	}

	c.nodeCache.Add(cacheKey, node, now)
	return
}

func (c *YpClient) Close() error {
	return c.clientConn.Close()
}
