package freshcache

import (
	"fmt"
	"path"
	"sync"
	"time"

	aLog "a.yandex-team.ru/library/go/core/log"

	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/log"
)

const (
	cacheCheckInterval = 60 * time.Second
)

type Cache struct {
	updateChan   chan CacheDriver
	updateTicker *time.Ticker
	stop         chan bool
	cachesPath   string
	drivers      []CacheDriver
	driversMutex sync.Mutex
	stats        map[string]*cacheDriverStat
	statsTicker  *time.Ticker
}

func (c *Cache) RegisterDriver(driver CacheDriver) {
	c.driversMutex.Lock()
	defer c.driversMutex.Unlock()
	c.drivers = append(c.drivers, driver)
	driverName := driver.GetOptions().Name
	c.stats[driverName] = newCacheDriverStat(driver)
}

func NewCache(cachesPath string) *Cache {
	return &Cache{
		updateChan:   make(chan CacheDriver),
		updateTicker: nil,
		stop:         make(chan bool),
		cachesPath:   cachesPath,
		drivers:      make([]CacheDriver, 0),
		driversMutex: sync.Mutex{},
		stats:        map[string]*cacheDriverStat{},
		statsTicker:  nil,
	}
}

func (c *Cache) Start() {
	for _, driver := range c.drivers {
		log.Logger.Info("loading driver cache", aLog.String("driver", driver.GetOptions().Name))
		if err := c.loadDriver(driver); err != nil {
			log.Logger.Error("failed to load driver cache", aLog.String("driver", driver.GetOptions().Name), aLog.Error(err))
			c.updateDriver(driver)
		}

		if DriverCacheExpired(driver) {
			c.updateDriver(driver)
		}
	}

	c.statsTicker = time.NewTicker(unistatTickInterval)
	c.updateTicker = time.NewTicker(cacheCheckInterval)

	go c.freshnessLoop()
	go c.updateLoop()
	go c.updateUnistatAges()
}

func (c *Cache) Stop() {
	c.statsTicker.Stop()
	c.updateTicker.Stop()
	c.stop <- true
	for _, drv := range c.drivers {
		drv.Stop()
		err := c.dumpDriver(drv)
		if err != nil {
			log.Logger.Error("failed to dump cache driver", aLog.String("driver", drv.GetOptions().Name), aLog.Error(err))
		}
	}
}

func (c *Cache) freshnessLoop() {
	for {
		select {
		case <-c.stop:
			return
		case <-c.updateTicker.C:
			log.Logger.Info("checking caches freshness")
			go c.checkFreshness()
		}
	}
}

func (c *Cache) checkFreshness() {
	c.driversMutex.Lock()
	defer c.driversMutex.Unlock()
	for _, driver := range c.drivers {
		options := driver.GetOptions()
		log.Logger.Info("checking driver", aLog.String("driver", options.Name))
		if DriverCacheExpired(driver) {
			log.Logger.Info("cache expired", aLog.String("driver", options.Name), aLog.Duration("cache_age", driver.GetAge()))
			c.updateChan <- driver
		} else {
			log.Logger.Debug("cache is fresh", aLog.String("driver", options.Name), aLog.Duration("cache_age", driver.GetAge()))
		}
	}
}

func (c *Cache) loadDriver(driver CacheDriver) error {
	return driver.LoadState(c.getCachePath(driver))
}

func (c *Cache) updateDriver(driver CacheDriver) {
	defer c.updateUnistatRefreshTime(driver, time.Now())
	driverName := driver.GetOptions().Name

	log.Logger.Info("updating driver cache", aLog.String("driver", driverName))
	err := driver.Update()
	if err != nil {
		log.Logger.Error("failed to update driver cache", aLog.String("driver", driverName), aLog.Error(err))
	}

	err = driver.DumpState(c.getCachePath(driver))
	if err != nil {
		log.Logger.Error("failed to dump driver cache", aLog.String("driver", driverName), aLog.Error(err))
	}
}

func (c *Cache) dumpDriver(driver CacheDriver) error {
	return driver.DumpState(c.getCachePath(driver))
}

func (c *Cache) updateLoop() {
	for {
		select {
		case <-c.stop:
			return
		case driver := <-c.updateChan:
			go c.updateDriver(driver)
		}
	}
}

func (c *Cache) getCachePath(driver CacheDriver) string {
	return path.Join(c.cachesPath, fmt.Sprintf("%s.data", driver.GetOptions().Name))
}
