package freshcache

import (
	"fmt"
	"math"
	"time"

	"a.yandex-team.ru/library/go/yandex/unistat"
	"a.yandex-team.ru/library/go/yandex/unistat/aggr"
)

const (
	ms                  = 1000
	s                   = 1000 * ms
	buckets             = 30
	unistatTickInterval = 5 * time.Second
)

var timeBuckets []float64

type cacheDriverStat struct {
	age         *unistat.Histogram
	refreshTime *unistat.Histogram
}

func init() {
	timeBuckets = make([]float64, buckets)
	for i := buckets - 1; i >= 0; i-- {
		timeBuckets[i] = math.Pow(1.5, float64(i-5)) * s
	}
}

func (c *Cache) updateUnistatAges() {
	for range c.statsTicker.C {
		for _, driver := range c.drivers {
			age := float64(driver.GetAge() / time.Microsecond)
			c.stats[driver.GetOptions().Name].age.Update(age)
		}
	}
}

func (c *Cache) updateUnistatRefreshTime(driver CacheDriver, now time.Time) {
	unistat.MeasureMicrosecondsSince(c.stats[driver.GetOptions().Name].refreshTime, now)
}

func newCacheDriverStat(driver CacheDriver) *cacheDriverStat {
	driverName := driver.GetOptions().Name
	stat := cacheDriverStat{
		age:         unistat.NewHistogram(fmt.Sprintf("fresh_cache_%s_age", driverName), 5, aggr.Histogram(), timeBuckets),
		refreshTime: unistat.NewHistogram(fmt.Sprintf("fresh_cache_%s_refresh_time", driverName), 5, aggr.Histogram(), timeBuckets),
	}

	unistat.Register(stat.age)
	unistat.Register(stat.refreshTime)

	return &stat
}
