package lru

import (
	"testing"
	"time"
)

type simpleStruct struct {
	int
	string
}

type complexStruct struct {
	int
	simpleStruct
}

var getTests = []struct {
	name       string
	keyToAdd   interface{}
	keyToGet   interface{}
	expectedOk bool
}{
	{"string_hit", "myKey", "myKey", true},
	{"string_miss", "myKey", "nonsense", false},
	{"simple_struct_hit", simpleStruct{1, "two"}, simpleStruct{1, "two"}, true},
	{"simple_struct_miss", simpleStruct{1, "two"}, simpleStruct{0, "noway"}, false},
	{"complex_struct_hit", complexStruct{1, simpleStruct{2, "three"}},
		complexStruct{1, simpleStruct{2, "three"}}, true},
}

func TestGet(t *testing.T) {
	deadline, _ := time.ParseDuration("10s")
	now := time.Now()

	for _, tt := range getTests {
		lru := New(0, deadline)
		lru.Add(tt.keyToAdd, 1234, now)
		val, ok := lru.Get(tt.keyToGet, now)
		if ok != tt.expectedOk {
			t.Fatalf("%s: cache hit = %v; want %v", tt.name, ok, !ok)
		} else if ok && val != 1234 {
			t.Fatalf("%s expected get to return 1234 but got %v", tt.name, val)
		}
	}
}

func TestRemove(t *testing.T) {
	deadline, _ := time.ParseDuration("10s")
	now := time.Now()

	lru := New(0, deadline)
	lru.Add("myKey", 1234, now)
	if val, ok := lru.Get("myKey", now); !ok {
		t.Fatal("TestRemove returned no match")
	} else if val != 1234 {
		t.Fatalf("TestRemove failed.  Expected %d, got %v", 1234, val)
	}

	lru.Remove("myKey")
	if _, ok := lru.Get("myKey", now); ok {
		t.Fatal("TestRemove returned a removed entry")
	}
}

func TestDeadline(t *testing.T) {
	deadline, _ := time.ParseDuration("10s")
	halfDeadline, _ := time.ParseDuration("5s")
	now := time.Now()

	lru := New(0, deadline)
	lru.Add("myKey", 1, now)
	if _, ok := lru.Get("myKey", now.Add(halfDeadline)); !ok {
		t.Fatal("TestRemove returned no match")
	}
	if _, ok := lru.Get("myKey", now.Add(deadline)); ok {
		t.Fatal("TestRemove returned match")
	}
	if _, ok := lru.Get("myKey", now); ok {
		t.Fatal("TestRemove returned match after deadline")
	}
}
