package util

import (
	"math/rand"
	"strings"
	"time"

	pb "a.yandex-team.ru/infra/rtc/instance_resolver/api"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/nanny"
	"a.yandex-team.ru/infra/rtc/instance_resolver/pkg/clients/qyp"
)

func GroupHostByIssCacher(hosts []*pb.THost) (result map[string][]string) {
	result = make(map[string][]string)
	for _, hostInfo := range hosts {
		for _, issCacher := range hostInfo.IssCachers {
			result[issCacher] = append(result[issCacher], hostInfo.HostName)
		}
	}
	return
}

func GroupHostByYpMaster(hosts []*pb.THost) (result map[string][]string) {
	result = make(map[string][]string)
	for _, hostInfo := range hosts {
		if hostInfo.YpMaster != "" {
			result[hostInfo.YpMaster] = append(result[hostInfo.YpMaster], hostInfo.HostName)
		}
	}
	return
}

func CreateNannyPersons(persons *nanny.NannyPersons) *pb.TNannyPersons {
	return &pb.TNannyPersons{
		Logins: persons.Logins,
		Groups: persons.Groups,
	}
}

func CreateQypOwners(owners *qyp.QypOwners) *pb.TQypOwners {
	return &pb.TQypOwners{
		Logins: owners.Logins,
		Groups: owners.Groups,
	}
}

func HasNannyServicesField(given []string) bool {
	if len(given) == 0 {
		return true
	}
	for _, field := range given {
		if strings.Split(field, ".")[0] == "NannyServices" {
			return true
		}
	}
	return false
}

func HasQloudEnvironmentsField(given []string) bool {
	if len(given) == 0 {
		return true
	}
	for _, field := range given {
		if strings.Split(field, ".")[0] == "QloudEnvironments" {
			return true
		}
	}
	return false
}

func HasQypVMsField(given []string) bool {
	if len(given) == 0 {
		return true
	}
	for _, field := range given {
		if strings.Split(field, ".")[0] == "QypVMs" {
			return true
		}
	}
	return false
}

func HasDeployDUsField(given []string) bool {
	if len(given) == 0 {
		return true
	}
	for _, field := range given {
		if strings.Split(field, ".")[0] == "DeployDUs" {
			return true
		}
	}
	return false
}

func GetNannyGroups(service *pb.TNannyService) (result []string) {
	// don't use ConfManagers
	result = make([]string, 0, len(service.Observers.Groups)+len(service.Owners.Groups)+len(service.OpsManagers.Groups))
	result = append(result, service.Observers.Groups...)
	result = append(result, service.Owners.Groups...)
	result = append(result, service.OpsManagers.Groups...)
	return
}

func GetNannyLogins(service *pb.TNannyService) (result []string) {
	// don't use ConfManagers
	result = make([]string, 0, len(service.Observers.Logins)+len(service.Owners.Logins)+len(service.OpsManagers.Logins))
	result = append(result, service.Observers.Logins...)
	result = append(result, service.Owners.Logins...)
	result = append(result, service.OpsManagers.Logins...)
	return
}

func GetQypGroups(vm *pb.TQypVM) (result []string) {
	return vm.Owners.Groups
}

func GetQypLogins(vm *pb.TQypVM) (result []string) {
	return vm.Owners.Logins
}

func GetQloudUsersAndGroups(service *pb.TQloudEnvironment) ([]string, []string) {
	users := make([]string, 0)
	groups := make([]string, 0)
	for _, qloudACE := range service.ACL.ACEs {
		if strings.EqualFold(qloudACE.Type, "user") {
			users = append(users, qloudACE.Id)
		} else {
			groups = append(groups, qloudACE.Id)
		}
	}
	return users, groups
}

// Retry backoff with jitter sleep to prevent overloaded conditions during intervals
// https://www.awsarchitectureblog.com/2015/03/backoff.html
func RetryBackoff(retry int, minBackoff, maxBackoff time.Duration) time.Duration {
	if retry < 0 {
		retry = 0
	}

	backoff := minBackoff << uint(retry)
	if backoff > maxBackoff || backoff < minBackoff {
		backoff = maxBackoff
	}

	if backoff == 0 {
		return 0
	}
	return time.Duration(rand.Int63n(int64(backoff)))
}

func ExtractYasmTags(tagString string) map[string]string {
	var res = make(map[string]string)
	var tags = strings.Split(tagString, " ")
	for _, tag := range tags {
		if strings.HasPrefix(tag, "a_") || strings.HasPrefix(tag, "itag_") {
			var tokens = strings.Split(tag, "_")
			if len(tokens) > 2 {
				res[tokens[1]] = tokens[2]
			}
		}
	}
	return res
}
