#include "main.h"

#include <library/cpp/getopt/opt.h>
#include <library/cpp/sighandler/async_signals_handler.h>
#include <library/cpp/logger/global/global.h>

#include <util/system/env.h>

#include <contrib/libs/grpc/include/grpc++/security/server_credentials.h>
#include <contrib/libs/grpc/include/grpc++/server_builder.h>
#include <contrib/libs/grpc/include/grpc++/server.h>
#include <contrib/libs/grpc/include/grpc++/resource_quota.h>

using namespace NLastGetopt;
using namespace NYP::NClient;

TDuration GetTimeout(grpc::ServerContext* context) {
    TDuration queryTimeout(TDuration::Seconds(15));
    auto deadline(std::chrono::time_point_cast<std::chrono::milliseconds>(context->deadline()));
    if (deadline.time_since_epoch().count()) {
        auto providedTimeout = deadline - std::chrono::system_clock::now();
        queryTimeout = TDuration::MicroSeconds(providedTimeout.count());
    }
    return queryTimeout;
}

void RunMain(int argc, char **argv) {
    InitGlobalLog2Console();

    TYpProxyOptions options;

    TOpts opts = TOpts::Default();

    opts.AddLongOption('d', "address", "bind to address for grpc server")
        .StoreResult(&options.Address).RequiredArgument().DefaultValue("localhost:9367");
    opts.AddLongOption('t', "token", "oauth token for YP, use YP_TOKEN by default")
        .StoreResult(&options.Token).RequiredArgument().DefaultValue(GetEnv("YP_TOKEN"));

    TOptsParseResult res(&opts, argc, argv);

    TYpProxyImpl proxyService(options);

    grpc::ResourceQuota quota;
    quota.SetMaxThreads(32);

    grpc::ServerBuilder builder;
    builder.AddListeningPort(options.Address, grpc::InsecureServerCredentials());
    builder.SetResourceQuota(quota);
    builder.RegisterService(&proxyService);

    std::unique_ptr<grpc::Server> server(builder.BuildAndStart());

    auto shutdownCallback = [&server](int) {
        server->Shutdown();
    };
    SetAsyncSignalFunction(SIGINT, shutdownCallback);
    SetAsyncSignalFunction(SIGTERM, shutdownCallback);

    INFO_LOG << "Started" << Endl;
    server->Wait();
    INFO_LOG << "Stopped" << Endl;
}

TYpProxyImpl::TYpProxyImpl(const TYpProxyOptions& options)
    : Token(options.Token)
{
}

grpc::Status TYpProxyImpl::FindPodsetsByNode(
        grpc::ServerContext* context,
        const api::TFindPodsetsByNodeRequest* request,
        api::TFindPodsetsByNodeReply* response) {

    TDuration queryTimeout(GetTimeout(context));

    auto ypClient(GetYpClient(request->GetOrigin()));
    if (!ypClient) {
        return grpc::Status(grpc::StatusCode::INVALID_ARGUMENT, "wrong origin provided");
    }

    // TODO: escaping
    try {
        auto queryResult = ypClient->SelectObjects<TPod>(
                {
                    "/meta/pod_set_id",
                    "/spec/iss",
                    "/status/dns/persistent_fqdn",
                    "/labels",
                    "/spec/host_infra/monitoring",
                    "/status/maintenance"
                },
                {
                    "[/spec/node_id] = '" + request->GetNodeName() + "'"
                }
            )
            .GetValue(queryTimeout);
        for (auto& item : queryResult.Results) {
            TPod pod;
            item.Fill(
                pod.MutableMeta()->mutable_pod_set_id(),
                pod.MutableSpec()->mutable_iss(),
                pod.MutableStatus()->mutable_dns()->mutable_persistent_fqdn(),
                pod.MutableLabels(),
                pod.MutableSpec()->mutable_host_infra()->mutable_monitoring(),
                pod.MutableStatus()->mutable_maintenance()
            );

            auto* instance(response->AddInstances());
            instance->SetYpPodSet(pod.Meta().pod_set_id());
            instance->SetContainerFqdn(pod.Status().dns().persistent_fqdn());

            {
                const auto& labels(pod.Labels().GetMapSafe());
                auto it(labels.find(TStringBuf("deploy_engine")));
                if (it != labels.end()) {
                    instance->SetDeployEngine(it->second.GetStringSafe());
                }
            }
            {
                const auto& labels(pod.Labels().GetMapSafe());
                auto it(labels.find(TStringBuf("nanny_service_id")));
                if (it != labels.end()) {
                    instance->SetNannyServiceId(it->second.GetStringSafe());
                }
            }

            for (const auto& issInstance : pod.Spec().iss().instances()) {
                for (const auto& prop : issInstance.properties()) {
                    if (prop.first == TStringBuf("tags") || prop.first == TStringBuf("all-tags")) {
                        instance->SetIssTags(prop.second);
                    }
                }
            }

            if (instance->GetDeployEngine() == TStringBuf("QYP")) {
                instance->SetFromQyp(true);
            } else if (instance->GetDeployEngine() == TStringBuf("YP_LITE")) {
                instance->SetFromNanny(true);
            } else if (instance->GetDeployEngine() == TStringBuf("MCRSC") || instance->GetDeployEngine() == TStringBuf("RSC")) {
                instance->SetFromDeploy(true);
            }

            if (instance->GetFromDeploy()) {
                const auto& labels(pod.Labels().GetMapSafe());
                {
                    auto it(labels.find(TStringBuf("environ")));
                    if (it != labels.end()) {
                        instance->SetDeployEnviron(it->second.GetStringSafe());
                    }
                }
                auto deployIt(labels.find(TStringBuf("deploy")));
                if (deployIt != labels.end()) {
                    const auto& deploy(deployIt->second.GetMapSafe());
                    {
                        auto it(deploy.find(TStringBuf("deploy_unit_id")));
                        if (it != labels.end()) {
                            instance->SetDeployUnitId(it->second.GetStringSafe());
                        }
                    }
                    {
                        auto it(deploy.find(TStringBuf("stage_id")));
                        if (it != labels.end()) {
                            instance->SetDeployStageId(it->second.GetStringSafe());
                        }
                    }
                    {
                        auto it(deploy.find(TStringBuf("project_id")));
                        if (it != labels.end()) {
                            instance->SetDeployProjectId(it->second.GetStringSafe());
                        }
                    }
                }
            }

            {
                const auto& labels(pod.Spec().host_infra().monitoring().labels());
                instance->MutableDeployMonitoring()->mutable_labels()->insert(labels.begin(), labels.end());
            }
            for (const auto& unistatEndpoint : pod.Spec().host_infra().monitoring().unistats()) {
                auto* workload(instance->MutableDeployMonitoring()->AddWorkloads());
                const auto& labels(unistatEndpoint.labels());
                workload->mutable_labels()->insert(labels.begin(), labels.end());
            }

            using NYP::NClient::NApi::NProto::EPodMaintenanceState;
            switch (pod.Status().maintenance().state()) {
                case EPodMaintenanceState::PMS_NONE:
                    instance->SetMaintenanceState("none");
                    break;
                case EPodMaintenanceState::PMS_REQUESTED:
                    instance->SetMaintenanceState("requested");
                    break;
                case EPodMaintenanceState::PMS_ACKNOWLEDGED:
                    instance->SetMaintenanceState("acknowledged");
                    break;
                case EPodMaintenanceState::PMS_IN_PROGRESS:
                    instance->SetMaintenanceState("in_progress");
                    break;
            }

        }
    } catch(...) {
        ERROR_LOG << "Request failed with " << CurrentExceptionMessage() << Endl;
        RemoveYpClient(request->GetOrigin());
        return grpc::Status(grpc::StatusCode::UNAVAILABLE, CurrentExceptionMessage());
    }

    INFO_LOG << response->GetInstances().size() << " instances found on " << request->GetNodeName() << Endl;

    return grpc::Status::OK;
}

grpc::Status TYpProxyImpl::GetNode(
        grpc::ServerContext* context,
        const api::TGetNodeRequest* request,
        api::TGetNodeReply* response) {

    TDuration queryTimeout(GetTimeout(context));

    auto ypClient(GetYpClient(request->GetOrigin()));
    if (!ypClient) {
        return grpc::Status(grpc::StatusCode::INVALID_ARGUMENT, "wrong origin provided");
    }

    // TODO: escaping
    try {
        auto queryResult = ypClient->GetObject<TNode>(
                request->GetNodeName(),
                {
                    "/meta/id",
                    "/labels",
                    "/status/maintenance"
                }
            )
            .GetValue(queryTimeout);

        TNode node;
        queryResult.Fill(
            node.MutableMeta()->mutable_id(),
            node.MutableLabels(),
            node.MutableStatus()->mutable_maintenance()
        );

        auto* result(response->MutableNode());
        result->SetNodeId(node.Meta().id());

        {
            const auto& labels(node.Labels().GetMapSafe());
            auto it(labels.find(TStringBuf("segment")));
            if (it != labels.end()) {
                result->SetSegment(it->second.GetStringSafe());
            }
        }

        using NYP::NClient::NApi::NProto::ENodeMaintenanceState;
        switch (node.Status().maintenance().state()) {
            case ENodeMaintenanceState::NMS_NONE:
                result->SetMaintenanceState("none");
                break;
            case ENodeMaintenanceState::NMS_REQUESTED:
                result->SetMaintenanceState("requested");
                break;
            case ENodeMaintenanceState::NMS_ACKNOWLEDGED:
                result->SetMaintenanceState("acknowledged");
                break;
            case ENodeMaintenanceState::NMS_IN_PROGRESS:
                result->SetMaintenanceState("in_progress");
                break;
        }

    } catch(...) {
        ERROR_LOG << "Request failed with " << CurrentExceptionMessage() << Endl;
        RemoveYpClient(request->GetOrigin());
        return grpc::Status(grpc::StatusCode::UNAVAILABLE, CurrentExceptionMessage());
    }

    INFO_LOG << "Node " << request->GetNodeName() << " found" << Endl;

    return grpc::Status::OK;
}

NYP::NClient::TClientPtr TYpProxyImpl::GetYpClient(const TString& origin) {
    TGuard<TAdaptiveLock> guard(Mutex);

    TYpClientMap::insert_ctx ctx;
    auto it = YpClients.find(origin, ctx);
    if (it != YpClients.end()) {
        return it->second;
    }

    INFO_LOG << "Creating client to " << origin << Endl;

    TClientOptions clientOptions;
    clientOptions.SetToken(Token);
    if (origin == TStringBuf("YP_SAS")) {
        clientOptions.SetAddress("sas.yp.yandex.net:8090");
    } else if (origin == TStringBuf("YP_SAS_TEST")) {
        clientOptions.SetAddress("sas-test.yp.yandex.net:8090");
    } else if (origin == TStringBuf("YP_MAN")) {
        clientOptions.SetAddress("man.yp.yandex.net:8090");
    } else if (origin == TStringBuf("YP_MAN_PRE")) {
        clientOptions.SetAddress("man-pre.yp.yandex.net:8090");
    } else if (origin == TStringBuf("YP_VLA")) {
        clientOptions.SetAddress("vla.yp.yandex.net:8090");
    } else if (origin == TStringBuf("YP_MYT")) {
        clientOptions.SetAddress("myt.yp.yandex.net:8090");
    } else if (origin == TStringBuf("YP_IVA")) {
        clientOptions.SetAddress("iva.yp.yandex.net:8090");
    } else {
        return nullptr;
    }

    it = YpClients.emplace_direct(ctx, origin, CreateClient(clientOptions));
    return it->second;
}

void TYpProxyImpl::RemoveYpClient(const TString& origin) {
    TGuard<TAdaptiveLock> guard(Mutex);

    auto it = YpClients.find(origin);
    if (it != YpClients.end()) {
        INFO_LOG << "Removing client to " << origin << Endl;
        YpClients.erase(it);
    }
}

int main(int argc, char** argv) noexcept {
    try {
        RunMain(argc, argv);
    } catch (...) {
        Cerr << CurrentExceptionMessage() << Endl;
        return 1;
    }
}
