# coding: utf-8

from __future__ import absolute_import, print_function

import logging
from datetime import timedelta

import click
from yaml import safe_load

from infra.rtc.iolimit_ticketer.cli import cli
from infra.rtc.iolimit_ticketer.cli_plan_it import PodDescriptor
import infra.rtc.iolimit_ticketer.yp_update as yp_update


@cli.command('apply_it')
@click.argument('plan', type=click.File('rb'))
@click.option('--sleep-time', type=int, default=30)
@click.option('--overwrite/--no-overwrite', default=False)
@click.option('--use-limits/--skip-limits', default=False)
@click.option('--zero-limits/--no-zero-limits', default=False)
@click.option('--services', default="")
@click.option('--force-yes/--no-force-yes', default=False)
@click.pass_context
def apply_it(ctx, plan, sleep_time, overwrite, use_limits, zero_limits, services, force_yes):
    """
    Apply plan.
    """
    if sleep_time <= 0:
        raise Exception("too fast")
    if zero_limits and not use_limits:
        raise Exception("zero limits can't be used without use limits")

    services = {str(x.strip()) for x in services.split(",") if x}

    plan_descriptor = safe_load(plan)
    cluster = plan_descriptor["cluster"]
    pods = []
    for pod in plan_descriptor["pods"]:
        pod = PodDescriptor.from_dict(pod)
        if services and pod.service_id not in services:
            continue
        pods.append(pod)

    pods.sort(key=lambda pod: (pod.account_id, pod.rack, pod.service_id, pod.pod_id))

    duration = timedelta(seconds=sleep_time * max(0, len(pods) - 1)) + timedelta(seconds=3 * sleep_time * max(0, len({pod.rack for pod in pods}) - 1))
    logging.info("overwrite: %r, use_limits: %r, zero_limits: %r, services: %r, force_yes: %r", overwrite, use_limits, zero_limits, services, force_yes)
    logging.info("cluster: %r, pod count: %r, sleep time: %d, duration: %s", cluster, len(pods), sleep_time, duration)

    if not ctx.obj.dry_run and not force_yes:
        click.confirm('Do you want to continue?', abort=True)

    yp_update.update_pods(cluster, pods, sleep_time,
                          dry_run=ctx.obj.dry_run, overwrite=overwrite,
                          use_limits=use_limits, zero_limits=zero_limits)
