
import logging

from yp.client import YpClient, find_token
import click
from infra.rtc.iolimit_ticketer.cli import cli


from infra.rtc.iolimit_ticketer.cli_apply_net_it import update_net_spec


INF_CAPACITY = 125000000000


def filter_pods_on_nodes(service_map, hosts):

    pods_on_hosts = {}

    for service_name, service_descriptor in service_map.items():

        # skipping tentacles, which does not have net quota
        if service_descriptor.account_id == "abc:service:31632":
            continue

        for dc, cluster in service_descriptor.clusters.items():
            for pod in cluster.pods.values():
                if pod.node_id in hosts:
                    if service_name not in pods_on_hosts:
                        pods_on_hosts[service_name] = []

                    pods_on_hosts[service_name].append(pod)

    return pods_on_hosts


def get_pod_guarantee(service_name, limits, debug=False):
    deploy_engine, service = service_name[0], service_name[1].split(".")[0]

    guarantee = limits[deploy_engine].get(service, {}).get("guarantee")

    if not guarantee:
        print("no guarantee for {0}".format(service))
        if debug is True:
            return 0
        else:
            return
    else:
        return guarantee


def change_host_net_capacity(yp_client, host, capacity, dry_run=True):

    bandwidth_path = "/spec/network/total_bandwidth"

    resource_id = yp_client.select_objects("resource",
                                           selectors=["/meta/id"],
                                           filter="[/meta/kind]='network' and [/meta/node_id]='{0}'".format(host),
                                           enable_structured_response=True
                                           )["results"][0][0]["value"]

    if dry_run is False:
        transaction_id = yp_client.start_transaction()
        yp_client.update_object(
            "resource",
            resource_id,
            set_updates=[{"path": bandwidth_path, "value": capacity}],
            transaction_id=transaction_id,
        )
        yp_client.commit_transaction(transaction_id)


def get_host_capacity(yp_client, host):

    return int(yp_client.select_objects(
        "resource",
        selectors=["/spec/network/total_bandwidth"],
        filter="[/meta/kind]='network' and [/meta/node_id]='{0}'".format(host)
    )[0][0])


@cli.command('apply_net_per_host')
@click.option('--hosts', type=str, default='', help='hosts to force guarantees, comma separated')
@click.option('--dc', type=str, default='')
@click.option('--overwrite/--no-overwrite', default=True)
@click.pass_context
def apply_net_guarantees_bad_nodes(ctx, hosts, dc, overwrite):

    hosts = set(hosts.split(","))
    if dc == "":
        dc = list(hosts)[0][0:3]

    service_map = ctx.obj.yp_stat.service_map
    filtered_data = filter_pods_on_nodes(service_map, hosts)

    yp_client = YpClient(dc, config=dict(token=find_token()))

    for host in list(hosts):

        current_capacity = get_host_capacity(yp_client, host)
        change_host_net_capacity(yp_client, host, INF_CAPACITY, dry_run=ctx.obj.dry_run)
        logging.info("Successfully increased capacity on {0}".format(host))

        for service, pod_descriptors in filtered_data.items():

            guarantee = get_pod_guarantee(service, ctx.obj.net_limit_map, debug=True)

            if not guarantee:
                continue

            for pod_descriptor in pod_descriptors:
                pod_id, node_id = pod_descriptor.pod_id, pod_descriptor.node_id

                if pod_descriptor.has_net_guarantee() is True:
                    logging.info("Pod {0} of {1} already has guarantee".format(pod_id, str(service[1])))
                    continue

                if node_id == host:
                    try:
                        update_net_spec(yp_client, pod_id, guarantee, 0, ctx.obj.dry_run, use_limits=False,
                                        overwrite=overwrite)
                        logging.info("Set guarantee {0} on pod {1}".format(str(guarantee), pod_id))
                    except Exception as err:
                        logging.exception("Pod {0} on service {1} failed to update with {2}".format(
                            pod_id, service[1], str(err))
                        )

        change_host_net_capacity(yp_client, host, current_capacity, dry_run=ctx.obj.dry_run)
        logging.info("Successfully decreased back capacity on {0}".format(host))
