# coding: utf-8

from __future__ import absolute_import, print_function


import logging
import time
from datetime import timedelta
from copy import copy

import click
from yaml import safe_load

from infra.rtc.iolimit_ticketer.cli import cli
from infra.rtc.iolimit_ticketer.cli_net_plan_it import PodDescriptor
import infra.rtc.iolimit_ticketer.yp_model as yp_model
from infra.rtc.iolimit_ticketer.yp_update import MEGABYTE


def check_if_fullhost_vm(yp_client, pod_id):

    resp = yp_client.select_objects(
        "pod", selectors=["/labels/qyp_vm_forced_node_id"], filter="[/meta/id]='{0}'".format(pod_id),
        enable_structured_response=True
    )["results"][0][0]["value"]

    if isinstance(resp, str):
        return True
    else:
        return False


def update_net_spec(yp_client, pod_id, guarantee, limit, dry_run, overwrite=False, use_limits=True):
    object_type = "pod"
    path = "/spec/resource_requests"
    resp = yp_client.get_object(
        object_type, pod_id, [path],
        options=dict(fetch_timestamps=True),
        enable_structured_response=True
    )

    write_data = False

    try:
        resource_requests = resp["result"][0]["value"]
        resource_requests_copy = copy(resource_requests)
        timestamp = resp["result"][0]["timestamp"]
    except:
        raise Exception("Incorrect pod spec")

    if overwrite is True:
        resource_requests["network_bandwidth_guarantee"] = guarantee * MEGABYTE
        resource_requests["network_bandwidth_limit"] = limit * MEGABYTE
    else:
        if not resource_requests.get("network_bandwidth_guarantee"):
            resource_requests["network_bandwidth_guarantee"] = guarantee * MEGABYTE
        if not resource_requests.get("network_bandwidth_limit"):
            resource_requests["network_bandwidth_limit"] = limit * MEGABYTE

    if use_limits is False:
        resource_requests["network_bandwidth_limit"] = 0

    guarantee_path, limit_path = path + "/network_bandwidth_guarantee", path + "/network_bandwidth_limit"

    if resource_requests != resource_requests_copy:
        if dry_run is False:
            transaction_id = yp_client.start_transaction()
            yp_client.update_object(
                object_type, pod_id,
                set_updates=[{"path": guarantee_path, "value": resource_requests["network_bandwidth_guarantee"]}],
                transaction_id=transaction_id,
                attribute_timestamp_prerequisites=[{"path": guarantee_path, "timestamp": timestamp}]
            )
            yp_client.update_object(
                object_type, pod_id,
                set_updates=[{"path": limit_path, "value": resource_requests["network_bandwidth_limit"]}],
                transaction_id=transaction_id,
                attribute_timestamp_prerequisites=[{"path": limit_path, "timestamp": timestamp}]
            )
            yp_client.commit_transaction(transaction_id)

            write_data = True

    return write_data


@cli.command('apply_net_it')
@click.argument('plan', type=click.File('rb'))
@click.option('--sleep-time', type=int, default=10)
@click.option('--overwrite/--no-overwrite', default=False)
@click.option('--use-limits/--skip-limits', default=False)
@click.option('--zero-limits/--no-zero-limits', default=False)
@click.option('--services', default="")
@click.option('--force-yes/--no-force-yes', default=False)
@click.pass_context
def apply_net_it(ctx, plan, sleep_time, overwrite, use_limits, zero_limits, services, force_yes):
    """
    Apply plan.
    """
    if sleep_time <= 0:
        raise Exception("too fast")
    if zero_limits and not use_limits:
        raise Exception("zero limits can't be used without use limits")

    services = {str(x.strip()) for x in services.split(",") if x}

    plan_descriptor = safe_load(plan)
    cluster = plan_descriptor["cluster"]
    pods = []
    for pod in plan_descriptor["pods"]:
        pod = PodDescriptor.from_dict(pod)
        if services and pod.service_id not in services:
            continue
        pods.append(pod)

    pods.sort(key=lambda pod: (pod.account_id, pod.rack, pod.service_id, pod.pod_id))

    duration = timedelta(seconds=sleep_time * max(0, len(pods) - 1)) + timedelta(seconds=3 * sleep_time * max(0, len({pod.rack for pod in pods}) - 1))
    logging.info("overwrite: %r, use_limits: %r, zero_limits: %r, services: %r, force_yes: %r", overwrite, use_limits, zero_limits, services, force_yes)
    logging.info("cluster: %r, pod count: %r, sleep time: %d, duration: %s", cluster, len(pods), sleep_time, duration)

    if not ctx.obj.dry_run and not force_yes:
        click.confirm('Do you want to continue?', abort=True)

    with yp_model.create_yp_client(yp_model.YP_ADDRESS_MAP[cluster]) as yp_client:
        prev_rack = None
        for pod in pods:

            if zero_limits:
                pod.network_limit = 0

            if prev_rack is not None and prev_rack != pod.rack:
                logging.info("Waiting before new rack...")
                time.sleep(sleep_time)
            prev_rack = pod.rack

            logging.info("Pod set id {0}".format(str(pod.pod_set_id)))
            if "bootstrap" in pod.pod_set_id:
                logging.info("Skipping samogon pod {0}".format(str(pod.pod_id)))
                continue

            if pod.deploy_engine == "QYP":
                if check_if_fullhost_vm(yp_client, pod.pod_id):
                    continue

            try:
                pod_spec_changed = update_net_spec(yp_client, pod.pod_id, pod.network_guarantee, pod.network_limit,
                                                   ctx.obj.dry_run,
                                                   overwrite=overwrite, use_limits=use_limits)

                if ctx.obj.dry_run is True:
                    logging.info("Pod %s/%s/%s from service %s spec should be changed: "
                                 "set limit to %s and guarantee to %s", pod.deploy_engine,
                                 pod.pod_id, pod.rack, pod.service_id, pod.network_limit, pod.network_guarantee)
                else:

                    if pod_spec_changed:

                        msg = "Pod {0}/{1}/{2} from service {3} spec has changed: set guarantee to {4}".format(
                            pod.deploy_engine, str(pod.pod_id), pod.rack, pod.service_id, str(pod.network_guarantee)
                        )
                        if use_limits:
                            msg += " and limits to {0}".format(str(pod.network_limit))

                        logging.info(msg)

            except Exception as err:
                logging.exception("Pod %s/%s/%s from service %s update failed: %s", pod.deploy_engine, pod.pod_id,
                                  pod.rack,
                                  pod.service_id, err)
