
from math import ceil
import logging
from collections import Counter, defaultdict

import click


from yp.common import YP_PRODUCTION_CLUSTERS
from infra.rtc.iolimit_ticketer.cli import cli
from infra.rtc.iolimit_ticketer.utils import MB_DIVISOR


def calculate_guarantees_to_increase(requred, available, total):

    total_quota_megabyte = total / MB_DIVISOR

    quota_to_add = (requred - available) / MB_DIVISOR

    if quota_to_add < 0:
        return 0
    else:
        total_quota_after_addition = total_quota_megabyte + quota_to_add

        if total_quota_after_addition <= 500:
            return ceil(quota_to_add / 100) * 100
        elif 5 * 1024 >= total_quota_after_addition > 500:
            return ceil(quota_to_add / 500) * 500
        elif 10 * 1024 >= total_quota_after_addition > 5 * 1024:
            return ceil(quota_to_add / 1024) * 1024
        else:
            return quota_to_add + ceil(total_quota_after_addition*0.1)


def calculate_guarantees_to_increase_by_cpu_ratio(guars):

    net_limit_mb = guars["limits"] / MB_DIVISOR

    if guars["cpu_usage"] * 4 > net_limit_mb:
        return int(((guars["cpu_usage"] * 4) - net_limit_mb))
    else:
        return 0


def create_resource_payload(net_bw_mb, account_id, cluster):
    return {
        "provider": "yp",
        "segment": cluster,
        "target_service": int(account_id.replace("abc:service:", "")),
        "target_folder": "default",
        "resource": "network_bandwidth",
        "amount": net_bw_mb * MB_DIVISOR,
        "units": "bytesPerSecond"
    }


@cli.command('calculate_abcd_quotas')
@click.option('--cluster', default="sas-test")
@click.option('--output_filename', default="")
@click.pass_context
def calculate_abcd_quotas(ctx, cluster, output_filename):

    total_volume_transfered = 0

    existing_guarantees_map = {}

    for clustr in YP_PRODUCTION_CLUSTERS:
        existing_guarantees_map[clustr] = defaultdict(Counter)

    accounts = ctx.obj.yp_stat.account_map

    for account, account_descriptor in accounts.items():

        default_resources = account_descriptor.clusters.get(cluster, {}).get("default")

        if default_resources:

            network_resources = default_resources.network
            existing_guarantees_map[cluster][account].update({"limits": network_resources.bandwidth_limits})
            existing_guarantees_map[cluster][account].update({"usage": network_resources.bandwidth_usage})

            cpu_resources = default_resources.cpu
            existing_guarantees_map[cluster][account].update({"cpu_limits": cpu_resources.cpu_limits})
            existing_guarantees_map[cluster][account].update({"cpu_usage": cpu_resources.cpu_usage})

    transfer = {
        "transfers": []
    }

    for account, account_data in existing_guarantees_map[cluster].items():

        if "abc:service" not in account:
            continue

        if account == "abc:service:31632":
            continue

        if account_data["cpu_limits"] == 0:
            continue

        quota_addition_value = calculate_guarantees_to_increase_by_cpu_ratio(account_data)

        if quota_addition_value == 0:
            continue

        payload = create_resource_payload(
            quota_addition_value, account, cluster
        )
        transfer["transfers"].append(payload)

    for record in transfer["transfers"]:
        total_volume_transfered += record["amount"]

    logging.info("For cluster {0} we have {1} Mb/s".format(cluster, str(total_volume_transfered / MB_DIVISOR)))

    if output_filename == "":
        output_filename = "transfer_net_bandwidth_{0}.txt".format(cluster)

    base_command_string = "./quota_mover --move --drop --provider yp-prod --segment {0} default --to-service {1} " \
                          "--resource network_bandwidth:{2}:bytesPerSecond\n"

    with open(output_filename, "w") as fle:
        for record in transfer["transfers"]:
            fle.write(base_command_string.format(cluster, str(record["target_service"]), str(int(record["amount"]))))
