# coding: utf-8

import json
import os
import requests
import math

import click
from infra.rtc.iolimit_ticketer.cli import cli

import infra.rtc.iolimit_ticketer.utils as utils

from yp.client import YpClient, find_token


def get_abc_group_id(servicename):
    token = utils.get_nanny_oauth_token()
    headers = {"Authorization": "OAuth {0}".format(token)}

    try:
        abc_id = requests.get(
            "https://nanny.yandex-team.ru/v2/services/{0}/info_attrs/".format(servicename),
            headers=headers
        ).json().get("content", {}).get("abc_group", None)
        return abc_id
    except:
        return None


def get_nannyservice_segment_cluster_location(servicename):
    clusters = ["iva", "myt", "vla", "sas", "man"]
    cluster_allocation = {}
    # добавляем сегмент в формате cluster_segment
    try:
        for cluster in clusters:
            with YpClient(cluster, config=dict(token=find_token())) as yp_client:
                result = yp_client.select_objects(
                    "pod",
                    selectors=["/labels/segment"],
                    filter="[/labels/nanny_service_id]=\"{0}\"".format(servicename),
                    enable_structured_response=True
                )
                try:
                    cluster_allocation["{0}".format(cluster)] = len(result["results"])
                except IndexError:
                    pass

    except Exception:
        for cluster in clusters:
            with YpClient(cluster, config=dict(token=find_token())) as yp_client:
                result = yp_client.select_objects(
                    "pod", selectors=["/labels/segment"],
                    filter="[/labels/nanny_service_id]=\"{0}\"".format(servicename),
                    enable_structured_response=True
                )
                try:
                    cluster_allocation["{0}".format(cluster)] = len(result["results"])
                except IndexError:
                    pass
    return cluster_allocation


@cli.command('change_limits')
@click.option('--nannyservices', default=None, type=str,
              help="Nanny services you want to increase limits, whitespace separated")
@click.option('--nannyservice_file', type=str, default=None)
@click.option('--ssd_limit', type=int, default=None)
@click.option('--hdd_limit', type=int, default=None)
@click.option('--overcommit_coefficient', type=int, default=None)
@click.option('--resource_type', type=str, default='io')
@click.pass_context
def change_limits(cntx, nannyservices, nannyservice_file, ssd_limit, hdd_limit, overcommit_coefficient, resource_type):
    nannyservice_list = []

    cntx.obj.read_tokens()

    if not nannyservices and not nannyservice_file:
        print("No nannyservices selected")
        return

    if not ssd_limit and not hdd_limit:
        print("No limits are going to be increased")
        return

    if ssd_limit > 500 or hdd_limit > 140:
        print("Limit is too large, please select valid value (500mb for ssd and 140mb for hdd)")

    if nannyservices:
        nannyservice_list.extend(nannyservices.split(" "))

    if nannyservice_file:
        with open(nannyservice_file) as fle:
            for line in fle:
                nannyservice_list.append(line.replace("\n", ""))

    limit_data = [i for i in json.load(open("io_limits.json")) if i["service_id"] in set(nannyservice_list)]

    quota_diff, updated_volumes = {}, {}

    for service in limit_data:

        if service.get("volumes"):

            ssd_volumes, hdd_volumes = set(), set()
            _ssd_guarantee, _hdd_guarantee, diff = 0, 0, {"ssd": None, "hdd": None}

            for vol in service["volumes"]:

                if vol["storage_class"] == "hdd":
                    _hdd_guarantee += vol["guarantee"]
                else:
                    _ssd_guarantee += vol["guarantee"]

                if vol["mount_path"] in ssd_volumes or vol["mount_path"] in hdd_volumes:
                    continue

                if vol["storage_class"] == "hdd":
                    hdd_volumes.add(vol["mount_path"])
                else:
                    ssd_volumes.add(vol["mount_path"])

            if hdd_limit:
                hdd_volumes = utils.split_quota(list(hdd_volumes), math.ceil(hdd_limit / 2), "hdd")
                service["volumes"] = [i for i in service["volumes"] if i["storage_class"] != "hdd"]

                for mp, guarant in hdd_volumes.items():
                    service["volumes"].append({"storage_class": "hdd", "mount_path": mp, "guarantee": guarant,
                                               "limit": guarant*2})
                diff["hdd"] = math.ceil(hdd_limit / 2) - _hdd_guarantee

            if ssd_volumes:
                ssd_volumes = utils.split_quota(list(ssd_volumes), ssd_limit, "ssd")
                service["volumes"] = [i for i in service["volumes"] if i["storage_class"] != "ssd"]

                for mp, guarant in ssd_volumes.items():
                    service["volumes"].append({"storage_class": "ssd", "mount_path": mp, "guarantee": guarant,
                                               "limit": guarant})

                diff["ssd"] = ssd_limit - _ssd_guarantee

            pod_allocation, abc_id = get_nannyservice_segment_cluster_location(service["service_id"]), \
                                       get_abc_group_id(service["service_id"])

            for location, podcount in pod_allocation.items():
                for storage_class, diff_value in diff.items():
                    if diff_value and diff_value > 0:
                        if podcount > 0:
                            if abc_id not in quota_diff:
                                quota_diff[abc_id] = [{"storage_class": storage_class, "quota": podcount * diff_value,
                                                       "cluster": location}]
                            else:
                                quota_present = False

                                for record in quota_diff[abc_id]:
                                    if record["storage_class"] == storage_class and record["cluster"] == location:
                                        record["quota"] += podcount * diff_value
                                        quota_present = True
                                        break

                                if quota_present is False:
                                    quota_diff[abc_id].append({
                                        "storage_class": storage_class,
                                        "quota": podcount * diff_value,
                                        "cluster": location
                                    })

            updated_volumes[service["service_id"]] = service["volumes"]

    final_filename = "diff_quotas.json"
    if os.path.isfile(final_filename) is True:
        counter = 1
        while True:
            filename = "diff_quotas_{0}.json".format(str(counter))

            if os.path.isfile(filename) is True:
                counter += 1

            else:
                json.dump(quota_diff, open(filename, "w"))
                final_filename = filename
                break
    else:
        json.dump(quota_diff, open(final_filename, "w"))

    print("saved quota to {}".format(final_filename))

    existing_limits = json.load(open("io_limits.json"))

    for lim in existing_limits:
        if lim["service_id"] in updated_volumes:
            lim["volumes"] = updated_volumes[lim["service_id"]]

    json.dump(existing_limits, open("io_limits.json", "w"), indent=4, sort_keys=True)

    print("limits have been changed successfully")
