# coding: utf-8

import json

from infra.rtc.iolimit_ticketer.cli import cli
import click


def calculate_qyp_limits(service_map):

    qyp_limit_objects = {}

    for key, service_stat in service_map.items():
        if key[0] == "QYP":

            for cluster, cluster_descr in service_stat.clusters.items():

                for pod, pod_descr in cluster_descr.pods.items():

                    assert pod_descr.segment_id in {"dev", "default"}

                    has_hdd, has_ssd = False, False

                    for volume in pod_descr.volume_desc_list:
                        if volume.storage_class == "ssd":
                            has_ssd = True
                        elif volume.storage_class == "hdd":
                            has_hdd = True

                    vcpu = pod_descr.resource_desc.vcpu_guarantee / 1000

                    if pod_descr.segment_id == "dev":
                        ssd_guarant, hdd_guarant = max(min(18*vcpu, 500), 18), max(min(3*vcpu, 70), 6)
                    else:
                        # для неразмеченных подов из дефолтного сегмента ставим общие дефолты
                        ssd_guarant, hdd_guarant = 30, 15

                    if not has_hdd and not has_ssd:
                        continue

                    limit_obj = {
                        "deploy_engine": "QYP",
                        "service_id": pod_descr.pod_id,
                        "volumes": [],
                        "segment": pod_descr.segment_id

                    }

                    if has_ssd:
                        limit_obj["volumes"].append(
                            {
                                "storage_class": "ssd", "guarantee": ssd_guarant, "limit": ssd_guarant
                            }
                        )
                    if has_hdd:
                        limit_obj["volumes"].append(
                            {
                                "storage_class": "hdd", "guarantee": hdd_guarant, "limit": hdd_guarant * 2
                            }
                        )

                    if pod_descr.pod_id not in qyp_limit_objects:
                        qyp_limit_objects[pod_descr.pod_id] = limit_obj

                    # если у вм более одного пода, то лимит/гарантию выбираем наибольшую
                    else:
                        for vol in qyp_limit_objects[pod_descr.pod_id]["volumes"]:
                            for new_vol in limit_obj["volumes"]:
                                if vol["storage_class"] == new_vol["storage_class"]:
                                    vol["guarantee"] = max(vol["guarantee"], new_vol["guarantee"])
                                    vol["limit"] = max(vol["limit"], new_vol["limit"])
                                    break

    return qyp_limit_objects


def merge_limits(service_map):

    limits_data = calculate_qyp_limits(service_map)

    io_limits = json.load(open("io_limits.json"))

    for record in io_limits:

        # Если вм в dev и уже размечена - меняем лимиты (поскольку это значит что мы ее разметили правилами для default)
        # Если вм в default и уже размечена - оставляем значения

        if record["deploy_engine"] != "QYP" or record["service_id"] not in limits_data:
            continue

        vm_data = limits_data.pop(record["service_id"])

        guarantees = {i["storage_class"]: i["guarantee"] for i in vm_data["volumes"]}
        record["volumes"] = [i for i in record["volumes"] if i["storage_class"] in guarantees]

        if vm_data["segment"] != "dev":
            continue

        for vol in record["volumes"]:
            vol["guarantee"] = guarantees[vol["storage_class"]]
            if vol["storage_class"] == "hdd":
                vol["limit"] = guarantees[vol["storage_class"]] * 2
            else:
                vol["limit"] = guarantees[vol["storage_class"]]

            guarantees.pop(vol["storage_class"])

        # возможны случаи пропущенных вольюмов
        for storage_class, guarantee in guarantees.items():
            record["volumes"].append(
                {
                    "storage_class": storage_class,
                    "guarantee": guarantee,
                    "limit": guarantee if storage_class == "ssd" else guarantee*2
                }
            )

    for vm_id, vm_data in limits_data.items():
        del vm_data["segment"]
        io_limits.append(vm_data)

    json.dump(io_limits, open("io_limits.json", "w"), indent=4, sort_keys=True)


@cli.command('compute_qyp_limits')
@click.pass_context
def compute_qyp_limits(ctx):
    merge_limits(ctx.obj.yp_stat.service_map)
