# coding: utf-8

from __future__ import absolute_import, print_function

import sys
import logging
from datetime import datetime

import click
from yaml import safe_dump
from dateutil.parser import parse as dt_parse

from infra.rtc.iolimit_ticketer.cli import cli
import infra.rtc.iolimit_ticketer.st_client as st_client
import infra.rtc.iolimit_ticketer.ok_client as ok_client
import infra.rtc.iolimit_ticketer.utils as utils


def find_expired_issues(service_map, ignore_approvement, storage_class):
    client = st_client.create_st_client()
    ticket_map = st_client.get_ticket_map(client, storage_class)
    grouped_services = utils.group_services_by_abc(service_map, target_storage_class=storage_class)
    for abc_service_id, local_services in grouped_services.items():
        if abc_service_id not in ticket_map or utils.have_all_services_io_limits(local_services, storage_class):
            continue

        issue_key = ticket_map[abc_service_id]
        issue = client.issues[issue_key]
        approvement = ok_client.get_approvement_from_issue(issue)
        if approvement is None:
            continue

        if ignore_approvement or approvement.is_expired() or approvement.is_rejected() or approvement.is_declined():
            if issue.deadline is None:
                yield issue_key
            elif dt_parse(issue.deadline) < datetime.now():
                yield issue_key


@cli.command()
@click.option('--ignore-approvement/--consider-approvement', default=False)
@click.option('--storage-class', default="ssd")
@click.pass_context
def expired(ctx, ignore_approvement, storage_class):
    """
    Show expired issues.
    """
    logging.info("ignore approvement: %r, storage class: %r", ignore_approvement, storage_class)

    issues = list(find_expired_issues(ctx.obj.yp_stat.service_map, ignore_approvement, storage_class))
    issues.sort()

    safe_dump({
        "issues": issues,
    }, sys.stdout, default_flow_style=False)
