# coding: utf-8

from __future__ import absolute_import, print_function

import sys
import logging
from math import ceil
from datetime import date

import click
from yaml import safe_dump

from infra.rtc.iolimit_ticketer.cli import cli
import infra.rtc.iolimit_ticketer.st_client as st_client
import infra.rtc.iolimit_ticketer.ok_client as ok_client
import infra.rtc.iolimit_ticketer.utils as utils
from yql.api.v1.client import YqlClient

from infra.rtc.iolimit_ticketer.yp_update import MEGABYTE
from infra.rtc.iolimit_ticketer.yp_model import _parse_account_id
from infra.rtc.iolimit_ticketer.cli_net_tickets import DEPLOY_DEPLOY_ENGINES


EXTRACT_NET_QUOTAS_QUERY = '''
SELECT total_data.account_id, NANVL(Cast(allocated_quota as Float) / Cast(total_quota as Float), 0.0) as quota_ratio,
    total_quota - allocated_quota as quota_diff
FROM
(SELECT
    `account_id`,
    sum(`net_bandwidth`) as allocated_quota
FROM hahn.`home/data_com/cubes/yp/yp_accounts/{0}`
where geo == '{1}' and quota_type == 'allocated_quota'
GROUP BY account_id) as allocated_data
join (
SELECT
    `account_id`,
    sum(`net_bandwidth`) as total_quota
FROM hahn.`home/data_com/cubes/yp/yp_accounts/{0}`
where geo == '{1}' and quota_type == 'total_quota'
GROUP BY account_id) as total_data on allocated_data.account_id == total_data.account_id
where total_data.account_id like 'abc:service%'
'''


ALL_DEPLOY_ENGINES = {"YP_LITE", "MCRSC", "RSC", "QYP"}


def find_services_ready_for_automation(service_map, ignore_approvement):
    client = st_client.create_st_client()
    ticket_map = st_client.get_ticket_map(client, "net")
    grouped_services = utils.group_services_by_abc(service_map)
    for abc_service_id, local_services in grouped_services.items():
        issue_key = ticket_map.get(abc_service_id)

        if not ignore_approvement:
            if not issue_key:
                continue
            issue = client.issues[issue_key]
            approvement = ok_client.get_approvement_from_issue(issue)
            if not approvement:
                continue
            if not approvement.is_approved():
                continue

        logging.info("ABC service %d (ticket %s) is ready for automation", abc_service_id, issue_key)
        for service_stat in local_services.values():
            yield service_stat


class PodDescriptor(object):

    def __init__(self):
        self.service_id = None
        self.pod_id = None
        self.account_id = None
        self.deploy_engine = None
        self.rack = None
        self.network_guarantee = None
        self.network_limit = None
        self.pod_set_id = None
        self.node_id = None

    def to_dict(self):
        return {
            "service_id": str(self.service_id),
            "pod_id": str(self.pod_id),
            "pod_set_id": str(self.pod_set_id),
            "account_id": str(self.account_id),
            "deploy_engine": str(self.deploy_engine),
            "rack": str(self.rack) if self.rack else None,
            "network_guarantee": self.network_guarantee,
            "network_limit": self.network_limit,
            "node_id": str(self.node_id)
        }

    @classmethod
    def from_dict(cls, data):
        pod_descriptor = cls()
        pod_descriptor.service_id = data["service_id"]
        pod_descriptor.pod_id = data["pod_id"]
        pod_descriptor.account_id = data["account_id"]
        pod_descriptor.rack = data["rack"]
        pod_descriptor.deploy_engine = data["deploy_engine"]
        pod_descriptor.network_guarantee = data["network_guarantee"]
        pod_descriptor.network_limit = data["network_limit"]
        pod_descriptor.pod_set_id = data["pod_set_id"]
        pod_descriptor.node_id = data.get("node_id")
        return pod_descriptor


def get_underquotted_services(cluster):
    client = YqlClient(db='hahn')

    abc_to_exclude = set()

    current_day = date.today().strftime("%Y-%m-%d")

    request = client.query(
        EXTRACT_NET_QUOTAS_QUERY.format(current_day, cluster),
        syntax_version=1
    )

    request.run()

    for table in request.get_results():
        table.fetch_full_data()

        for row in table.rows:
            if row[1] >= 0.95 or row[2] <= 50 * MEGABYTE:
                abc_to_exclude.add(_parse_account_id(row[0]))

    return abc_to_exclude


def create_net_plan_it(ctx, ignore_approvement, cluster, deploy_engines, abc_services, ignore_abc_services,
                       consider_underquotted, services, cpu_base, output_pipeline):
    """
    Generate net plan.
    """
    deploy_engines = {x.strip() for x in deploy_engines.split(",") if x} or {"YP_LITE"}
    abc_services = {int(x.strip()) for x in abc_services.split(",") if x}
    ignore_abc_services = {int(x.strip()) for x in ignore_abc_services.split(",") if x}
    logging.info("ignore approvement: %r, cluster: %r, deploy engines: %r, abc services: %r, ignore abc service: %r",
                 ignore_approvement, cluster, deploy_engines, abc_services, ignore_abc_services)
    assert ALL_DEPLOY_ENGINES.issuperset(deploy_engines), "wrong deploy engines given"

    pods = []
    pods_scanned = 0
    pods_without_limits = 0
    pods_from_unknown_deploy_engine = 0

    if consider_underquotted is True:
        underquotted_services = get_underquotted_services(cluster)
        ignore_abc_services = set.union(ignore_abc_services, underquotted_services)

    for service_stat in find_services_ready_for_automation(ctx.obj.yp_stat.service_map, ignore_approvement):
        if abc_services and service_stat.get_abc_id() not in abc_services:
            continue
        if service_stat.get_abc_id() in ignore_abc_services:
            continue
        if cluster not in service_stat.clusters:
            continue

        if service_stat.deploy_engine in DEPLOY_DEPLOY_ENGINES:
            service_id = service_stat.service_id.split(".")[0]
        else:
            service_id = service_stat.service_id

        limit_map = ctx.obj.net_limit_map.get(service_stat.deploy_engine, {}).get(service_id, {})

        cluster_stat = service_stat.clusters[cluster]
        for pod_id, pod_stat in cluster_stat.pods.items():
            pod_descriptor = PodDescriptor()
            pod_descriptor.service_id = service_stat.service_id

            if services and pod_descriptor.service_id not in services:
                if pod_descriptor.service_id not in services:
                    continue

            pod_descriptor.pod_id = pod_id
            pod_descriptor.pod_set_id = pod_stat.pod_set_id
            pod_descriptor.account_id = service_stat.account_id
            pod_descriptor.deploy_engine = service_stat.deploy_engine

            node_descriptor = ctx.obj.yp_stat.node_map.get(pod_stat.node_id)
            if node_descriptor is not None:
                pod_descriptor.rack = node_descriptor.rack
                pod_descriptor.node_id = node_descriptor.fqdn

            if service_stat.deploy_engine not in deploy_engines:
                pods_from_unknown_deploy_engine += 1
                continue

            elif service_stat.deploy_engine in deploy_engines:

                if cpu_base is True:
                    pod_descriptor.network_guarantee = ceil(pod_stat.resource_desc.vcpu_guarantee / 1000) * 4
                    pod_descriptor.network_limit = ceil((pod_stat.resource_desc.vcpu_guarantee / 1000) * 1.5) * 4

                else:
                    if not pod_stat.resource_desc.network_bandwidth_guarantee:
                        if limit_map:
                            pod_descriptor.network_guarantee = limit_map["guarantee"]
                            pod_descriptor.network_limit = limit_map["limit"]
                        else:
                            logging.warning("Pod %s/%s has no guarantee, applying default", service_stat.service_id, pod_id)
                            pod_descriptor.network_guarantee = 10
                            pod_descriptor.network_limit = 15

            else:
                pods_from_unknown_deploy_engine += 1

            pods_scanned += 1
            if pod_descriptor.network_guarantee or pod_descriptor.network_limit:
                pods.append(pod_descriptor)

    logging.info("Pods found: %d, pods without guarantees: %d, pods from unknown deploy engine: %d, pods scanned: %d",
                 len(pods), pods_without_limits, pods_from_unknown_deploy_engine, pods_scanned)

    if output_pipeline == "stdout":
        safe_dump({
            "pods": [pod.to_dict() for pod in pods],
            "cluster": cluster
        }, sys.stdout, default_flow_style=False)
    elif output_pipeline == "dict":
        return {
            "pods": [pod.to_dict() for pod in pods],
            "cluster": cluster
        }
    else:
        raise NotImplementedError("Unsupported way of processing plan output")


@cli.command('net_plan_it')
@click.option('--ignore-approvement/--consider-approvement', default=False)
@click.option('--cluster', default="sas-test")
@click.option('--deploy-engines', default="")
@click.option('--abc-services', default="")
@click.option('--ignore-abc-services', default="")
@click.option('--consider-underquotted/--skip-underquotted', default=True)
@click.option('--services', default=None)
@click.option('--cpu-base/--map_base', default=False)
@click.pass_context
def net_plan_it(ctx, ignore_approvement, cluster, deploy_engines, abc_services, ignore_abc_services,
                consider_underquotted, services, cpu_base):

    services_set = None
    if services:
        services_set = set()

        with open(services) as fle:
            for line in fle:
                services_set.add(line.strip("\n"))

    create_net_plan_it(
        ctx, ignore_approvement, cluster, deploy_engines, abc_services, ignore_abc_services,
                       consider_underquotted, services_set, cpu_base, output_pipeline="stdout"
    )
