# coding: utf-8

from __future__ import absolute_import, print_function

import difflib
import logging
from collections import defaultdict

import click

from nanny_repo import repo_pb2
from infra.rtc.iolimit_ticketer.cli import cli
import infra.rtc.iolimit_ticketer.st_client as st_client
import infra.rtc.iolimit_ticketer.nanny_client as nanny_client
import infra.rtc.iolimit_ticketer.ok_client as ok_client
import infra.rtc.iolimit_ticketer.utils as utils

NEWLINE = u"\n"
DEFAULT_DEPLOY_ENGINES = {"YP_LITE"}
DEPLOY_DEPLOY_ENGINES = {"MCRSC", "RSC"}
QYP_DEPLOY_ENGINES = {"QYP"}
ALL_DEPLOY_ENGINES = DEFAULT_DEPLOY_ENGINES | DEPLOY_DEPLOY_ENGINES | QYP_DEPLOY_ENGINES
NANNY_NOTIFICATION_PREFIX = "net-limit-"


def build_nanny_net_cgi(net_guarantee):
    parts = ["netGuarantee={}".format(net_guarantee), "netLimit=0"]
    if not net_guarantee:
        return True, parts
    else:
        return False, parts


def build_yasm_url(pod_stat, cluster_stat):
    if pod_stat.yasm_tags:
        geo = cluster_stat.cluster_name
        if geo in ("iva", "myt"):
            geo = "msk"
        return "https://yasm.yandex-team.ru/template/panel/Porto-container/hosts=ASEARCH;itype={itype};ctype={ctype};prj={prj};geo={geo}".format(
            itype=pod_stat.yasm_tags["itype"],
            ctype=pod_stat.yasm_tags["ctype"],
            prj=pod_stat.yasm_tags["prj"],
            geo=geo
        )
    else:
        return None


# Нужно для того чтоб обновить тикеты "про лимиты" на тикеты "про гарантии"
def get_updated_summary_net(abc_name):
    return u'Включить NET гарантии для сервисов {}'.format(abc_name.get("en") or abc_name.get("ru"))


def filter_dev_pods(yp_stat):

    for service, service_descriptor in yp_stat.items():
        for cluster_name, cluster_descriptor in service_descriptor.clusters.items():
            for pod_name, pod_descriptor in cluster_descriptor.pods.items():
                if pod_descriptor.segment_id == "dev":
                    try:
                        del yp_stat[service]
                    except KeyError:
                        pass

    return yp_stat


def update_net_description(client, issue, abc_service_id, abc_members, abc_services, yp_services, net_limit_map,
                           nanny_modifier, dry_run=True, change_summary=False):
    abc_name, abc_slug = abc_services[abc_service_id]

    description = []
    description.append(utils.render_template_from_resource(
        "/net_header.txt",
        abc_slug=abc_slug,
        abc_name=abc_name.get("en") or abc_name.get("ru")
    ))
    description.append(NEWLINE)

    yp_services_map = defaultdict(list)
    for (deploy_engine, service_id), service_stat in yp_services.items():
        has_net_guarantee = service_stat.has_net_guarantee()
        yp_services_map[(deploy_engine, has_net_guarantee)].append(service_stat)

    has_not_equality = False

    service_counter = 0
    service_without_limit_counter = 0

    for key, service_list in sorted(yp_services_map.items()):
        deploy_engine, has_net_guarantee = key

        template_service_list = []
        for service_stat in sorted(service_list, key=lambda x: x.service_id):  # type: yp_model.ServiceDescriptor
            service_counter += 1
            if deploy_engine == "YP_LITE":
                net_guarantee = 0
                net_limit = 0
                net_spec = net_limit_map.get(deploy_engine, {}).get(service_stat.service_id, {})
                if net_spec:
                    net_guarantee = net_spec["guarantee"] or 0
                    net_limit = 0

                has_no_limit_overall = False
                template_cluster_list = []
                for cluster_stat in sorted(service_stat.clusters.values(), key=lambda x: x.cluster_name):
                    first_pod = cluster_stat.get_first_pod_id()
                    if not first_pod:
                        continue

                    has_no_limit, net_cgi = build_nanny_net_cgi(net_guarantee)
                    has_no_limit_overall = has_no_limit_overall or has_no_limit

                    has_different_pods = len({pod_stat.resource_desc.to_tuple() for pod_stat in cluster_stat.pods.values()}) != 1
                    if has_different_pods:
                        has_not_equality = True

                    template_cluster_list.append({
                        "has_guarantee_and_limit": cluster_stat.has_net_guarantee(),
                        "has_different_pods": has_different_pods,
                        "first_pod": first_pod,
                        "cluster_name": cluster_stat.cluster_name,
                        "net_cgi": "&".join(net_cgi),
                        "yasm_url": build_yasm_url(cluster_stat.pods[first_pod], cluster_stat)
                    })

                if has_no_limit_overall:
                    service_without_limit_counter += 1

                template_service = {
                    "service_id": service_stat.service_id,
                    "clusters": template_cluster_list,
                    "has_no_limit_overall": has_no_limit_overall,
                    "net_guarantee": net_guarantee,
                    "net_limit": net_limit
                }

            elif deploy_engine in DEPLOY_DEPLOY_ENGINES:

                if "." in service_stat.service_id:
                    stage = service_stat.service_id.split(".")[0]
                else:
                    stage = service_stat.service_id

                net_guarantee = net_limit_map.get(deploy_engine, {}).get(stage, {}).get("guarantee")

                if not net_guarantee:
                    all_deploy_engines = list(DEPLOY_DEPLOY_ENGINES)

                    if deploy_engine == all_deploy_engines[0]:
                        deploy_engine = all_deploy_engines[1]
                    else:
                        deploy_engine = all_deploy_engines[0]

                    net_guarantee = net_limit_map.get(deploy_engine, {}).get(stage, {}).get("guarantee")

                if not net_guarantee:
                    service_without_limit_counter += 1
                template_service = {
                    "deploy_unit_id": service_stat.deploy_unit_id,
                    "stage_id": service_stat.deploy_stage_id,
                    "has_guarantee_and_limit": service_stat.has_net_guarantee_and_limit(),
                    "net_guarantee": net_guarantee
                }

            elif deploy_engine == "QYP":
                net_guarantee = net_limit_map.get(deploy_engine, {}).get(service_stat.service_id, {}).get("guarantee")
                if not net_guarantee:
                    service_without_limit_counter += 1
                clusters = list(service_stat.clusters.keys())

                if len(clusters) == 1:
                    template_service = {
                        "cluster": clusters[0],
                        "vm_id": service_stat.service_id,
                        "has_guarantee_and_limit": service_stat.has_net_guarantee_and_limit(),
                        "net_guarantee": net_guarantee
                    }
                else:
                    template_service = None

            else:
                raise Exception("{} not supported".format(deploy_engine))

            if template_service:
                template_service_list.append(template_service)

        description.append(utils.render_template_from_resource(
            '/net_service_list_{}.txt'.format(deploy_engine.lower()),
            deploy_engine=deploy_engine,
            has_net_limit=has_net_guarantee,
            service_list=template_service_list
        ))
        description.append(NEWLINE)

    description.append(utils.render_template_from_resource(
        '/net_footer.txt',
        has_not_equality=has_not_equality,
        service_without_limit_counter=service_without_limit_counter
    ))
    description.append(NEWLINE)

    description = u"".join(description)
    if issue.description != description:
        if dry_run:
            textdiff = u"\n".join(difflib.unified_diff(issue.description.splitlines(), description.splitlines(), n=1))
            logging.warning("Ticket %r for abc service %d should be updated:\n %s\n", issue, abc_service_id, textdiff)
        else:
            issue.update(description=description)

            if change_summary:
                issue.update(summary=get_updated_summary_net(abc_name))

    logging.info("ABC %d, services found %d, services without modelled limit %d", abc_service_id, service_counter,
                 service_without_limit_counter)


def update_net_st_tickets(service_map, abc_members, abc_services, abc_responsibles, net_limit_map,
                          dry_run, use_approvement, target_deploy_engines,
                          target_abc_services, ignore_abc_services, ignore_activity,
                          update_closed, change_summary, remove_notifics):
    """
    :type service_map: dict[(str, str), yp_model.ServiceDescriptor]
    """
    client = st_client.create_st_client()
    ticket_map = st_client.get_ticket_map(client, "net", get_closed=update_closed)

    nanny_modifier = nanny_client.NannyServiceModifier(
        template_name="/nanny_net.txt",
        prefix=NANNY_NOTIFICATION_PREFIX,
        dry_run=dry_run,
        severity_level=repo_pb2.PerServiceNotification.WARNING
    )

    if remove_notifics:
        logging.info("Removing ALL nanny notifications")
        nanny_modifier.remove_all_notifications()
        return

    grouped_services = utils.group_services_by_abc(service_map, target_deploy_engines=target_deploy_engines)
    for abc_service_id, local_services in grouped_services.items():

        local_services = {k: v for k, v in local_services.items()}

        if target_abc_services and abc_service_id not in target_abc_services:
            continue
        if abc_service_id in ignore_abc_services:
            continue

        if utils.have_all_services_net_guarantees(local_services) and abc_service_id not in ticket_map:
            # services has limits, don't create new ticket for them
            continue

        if abc_service_id not in abc_services or abc_service_id not in abc_members:
            logging.warning("Abc service %d is incomplete", abc_service_id)
            continue

        persons = {login for _, login in abc_members[abc_service_id]}
        try:
            persons.update(login for _, login in abc_responsibles[abc_service_id])
        except KeyError:
            persons.add("glebskvortsov")

        persons = sorted(persons)

        issue_key = st_client.get_or_create_ticket(client, ticket_map, abc_service_id, abc_services, "net", persons,
                                                   dry_run=dry_run)
        if issue_key is None:
            logging.warning("Ticket for abc service %d should be created", abc_service_id)
            continue

        issue = client.issues[issue_key]
        update_net_description(
            client=client,
            issue=issue,
            abc_service_id=abc_service_id,
            abc_members=abc_members,
            abc_services=abc_services,
            yp_services=local_services,
            net_limit_map=net_limit_map,
            nanny_modifier=nanny_modifier,
            dry_run=dry_run,
            change_summary=change_summary,
        )

        if utils.have_all_services_net_guarantees(local_services):
            if dry_run:
                logging.warning("Ticket %r for abc service %d should be closed as ready", issue, abc_service_id)
            else:
                try:
                    transition = issue.transitions['close']
                    transition.execute(resolution='fixed', comment=u'Сетевые гарантии применены, спасибо!')
                except:
                    # невозможность совершить переход означает что тикет уже закрыт и мы только меняем его описание
                    pass

        elif use_approvement and ok_client.get_approvement_id_from_issue(issue) is None:
            text = utils.render_template_from_resource('/net_approvement.txt')

            has_job = False
            for service_stat in local_services.values():
                if service_stat.deploy_engine == "YP_LITE" and not service_stat.has_net_guarantee_and_limit():
                    has_job = True

            if has_job and (not st_client.has_activity(issue) or ignore_activity):
                if dry_run:
                    logging.warning("Approvement %r for abc service %d should be created", issue, abc_service_id)
                else:
                    assert persons
                    ok_client.post_approvement_to_issue(issue, text, persons)

    if not target_abc_services and not ignore_abc_services:
        logging.info("Closing unrelevant tickets")
        all_services = utils.group_services_by_abc(service_map)
        st_client.close_unrelevant_tickets(client, ticket_map, all_services, dry_run=dry_run)
    else:
        logging.warning("Not closing unrelevant tickets")

    if not target_abc_services and not ignore_abc_services and "YP_LITE" in target_deploy_engines:
        logging.info("Cleanup nanny notifications")
        nanny_modifier.cleanup()
    else:
        logging.warning("Not updating nanny notifications")


@cli.command('net_tickets')
@click.option('--use-approvement/--no-approvement', default=False)
@click.option('--ignore-activity/--consider-activity', default=False)
@click.option('--deploy-engines', default="")
@click.option('--abc-services', default="")
@click.option('--ignore-abc-services', default="")
@click.option('--update-closed/--not-update-closed', default=False)
@click.option('--change-summary/--not-change-summary', default=False)
@click.option('--remove-nanny-notif/--not-remove-nanny-notif', default=False)
@click.option('--process-dev/--ignore-dev', default=False)
@click.pass_context
def net_tickets(ctx, use_approvement, ignore_activity, deploy_engines, abc_services, ignore_abc_services,
                update_closed, change_summary, remove_nanny_notif, process_dev):
    """
    Generate net tickets.
    """
    deploy_engines = {x.strip() for x in deploy_engines.split(",") if x} or ALL_DEPLOY_ENGINES
    abc_services = {int(x.strip()) for x in abc_services.split(",") if x}
    ignore_abc_services = {int(x.strip()) for x in ignore_abc_services.split(",") if x}
    ignore_abc_services.add(31632)  # tentacles
    ignore_abc_services.add(30776)  # tentacles too
    logging.info("use approvement: %r, ignore activity: %r, deploy engines: %r, abc services: %r, ignore abc services: %r",
                 use_approvement, ignore_activity, deploy_engines, abc_services, ignore_abc_services)
    assert ALL_DEPLOY_ENGINES.issuperset(deploy_engines), "wrong deploy engines given"

    if not ctx.obj.dry_run:
        click.confirm('Do you want to continue?', abort=True)

    if not process_dev:
        service_map = filter_dev_pods(ctx.obj.yp_stat.service_map)
    else:
        service_map = ctx.obj.yp_stat.service_map

    update_net_st_tickets(
        service_map=service_map,
        abc_members=ctx.obj.abc_members,
        abc_services=ctx.obj.abc_services,
        abc_responsibles=ctx.obj.abc_responsibles,
        net_limit_map=ctx.obj.net_limit_map,
        dry_run=ctx.obj.dry_run,
        use_approvement=use_approvement,
        target_deploy_engines=deploy_engines,
        target_abc_services=abc_services,
        ignore_abc_services=ignore_abc_services,
        ignore_activity=ignore_activity,
        update_closed=update_closed,
        change_summary=change_summary,
        remove_notifics=remove_nanny_notif
    )
