# coding: utf-8

from __future__ import absolute_import, print_function

import math
import sys
import logging

import click
from yaml import safe_dump

from infra.rtc.iolimit_ticketer.cli import cli
import infra.rtc.iolimit_ticketer.st_client as st_client
import infra.rtc.iolimit_ticketer.ok_client as ok_client
import infra.rtc.iolimit_ticketer.utils as utils

QYP_MIN_ROOT_FS_LIMIT = 10
ALL_DEPLOY_ENGINES = {"YP_LITE", "MCRSC", "RSC", "QYP"}


def compute_qyp_bandwidth(total_bandwidth, volumes):
    root_fs_bandwidth = 0
    disk_count = 0
    for spec in volumes:
        if spec.mount_path == "/":
            root_fs_bandwidth = min(total_bandwidth / 3, QYP_MIN_ROOT_FS_LIMIT)
        else:
            disk_count += 1
    total_bandwidth -= root_fs_bandwidth
    one_volume_bandwidth = total_bandwidth
    if disk_count:
        one_volume_bandwidth /= disk_count
    return root_fs_bandwidth, one_volume_bandwidth


def check_volume_stats(service_name, deploy_engine, pod_limits_map, volume_descriptor):

    iolimits_data = pod_limits_map[deploy_engine].get(service_name)

    if not iolimits_data:
        # игнорируем сервисы, которых нет в io_limits
        return True

    if "net" in iolimits_data:
        iolimits_data.pop("net")

    if len(iolimits_data) == 0:
        return True

    try:
        iolimits_limit = iolimits_data[(volume_descriptor.mount_path,
                                        volume_descriptor.storage_class)]["limit"] * 1024 * 1024
        iolimits_guarantee = iolimits_data[(volume_descriptor.mount_path,
                                            volume_descriptor.storage_class)]["guarantee"] * 1024 * 1024
    except KeyError:
        # игнорируем вольюмы, про которые io_limits не знает вообще
        return True

    if iolimits_limit != volume_descriptor.bandwidth_limit or \
       iolimits_guarantee != volume_descriptor.bandwidth_guarantee:
        return False

    return True


def find_services_ready_for_automation(service_map, ignore_approvement, storage_class):
    client = st_client.create_st_client()
    ticket_map = st_client.get_ticket_map(client, storage_class)
    grouped_services = utils.group_services_by_abc(service_map, target_storage_class=storage_class)
    for abc_service_id, local_services in grouped_services.items():
        issue_key = ticket_map.get(abc_service_id)

        if not ignore_approvement:
            if not issue_key:
                continue
            issue = client.issues[issue_key]
            approvement = ok_client.get_approvement_from_issue(issue)
            if not approvement:
                continue
            if not approvement.is_approved():
                continue

        logging.info("ABC service %d (ticket %s) is ready for automation", abc_service_id, issue_key)
        for service_stat in local_services.values():
            yield service_stat


class VolumeDescriptor:

    def __init__(self):
        self.mount_path = None
        self.bandwidth_limit = None
        self.bandwidth_guarantee = None
        self.storage_class = None

    def to_dict(self):
        return {
            "mount_path": str(self.mount_path),
            "bandwidth_limit": self.bandwidth_limit,
            "bandwidth_guarantee": self.bandwidth_guarantee,
            "storage_class": self.storage_class
        }

    @classmethod
    def from_dict(cls, data):
        volume_descriptor = cls()
        volume_descriptor.mount_path = data["mount_path"]
        volume_descriptor.bandwidth_limit = data["bandwidth_limit"]
        volume_descriptor.bandwidth_guarantee = data["bandwidth_guarantee"]
        volume_descriptor.storage_class = data["storage_class"]
        return volume_descriptor


class PodDescriptor:

    def __init__(self):
        self.service_id = None
        self.pod_id = None
        self.account_id = None
        self.deploy_engine = None
        self.rack = None
        self.volumes = []

    def to_dict(self):
        return {
            "service_id": str(self.service_id),
            "pod_id": str(self.pod_id),
            "account_id": str(self.account_id),
            "deploy_engine": str(self.deploy_engine),
            "rack": str(self.rack) if self.rack else None,
            "volumes": [volume.to_dict() for volume in self.volumes]
        }

    @classmethod
    def from_dict(cls, data):
        pod_descriptor = cls()
        pod_descriptor.service_id = data["service_id"]
        pod_descriptor.pod_id = data["pod_id"]
        pod_descriptor.account_id = data["account_id"]
        pod_descriptor.rack = data["rack"]
        pod_descriptor.deploy_engine = data["deploy_engine"]
        for volume in data["volumes"]:
            pod_descriptor.volumes.append(VolumeDescriptor.from_dict(volume))
        return pod_descriptor


@cli.command('plan_it')
@click.option('--ignore-approvement/--consider-approvement', default=False)
@click.option('--cluster', default="sas-test")
@click.option('--deploy-engines', default="")
@click.option('--abc-services', default="")
@click.option('--ignore-abc-services', default="")
@click.option('--ignore-pod-spec/--consider-pod-spec', default=False)
@click.option('--storage-class', default="ssd")
@click.pass_context
def plan_it(ctx, ignore_approvement, cluster, deploy_engines, abc_services, ignore_abc_services, storage_class,
            ignore_pod_spec):
    """
    Generate plan.
    """
    deploy_engines = {x.strip() for x in deploy_engines.split(",") if x} or {"YP_LITE"}
    abc_services = {int(x.strip()) for x in abc_services.split(",") if x}
    ignore_abc_services = {int(x.strip()) for x in ignore_abc_services.split(",") if x}
    logging.info("ignore approvement: %r, cluster: %r, deploy engines: %r, abc services: %r, ignore abc service: %r, storage class: %r",
                 ignore_approvement, cluster, deploy_engines, abc_services, ignore_abc_services, storage_class)
    assert ALL_DEPLOY_ENGINES.issuperset(deploy_engines), "wrong deploy engines given"

    pods = []
    pods_scanned = 0
    pods_without_limits = 0
    pods_from_unknown_deploy_engine = 0
    for service_stat in find_services_ready_for_automation(ctx.obj.yp_stat.service_map, ignore_approvement, storage_class):
        if abc_services and service_stat.get_abc_id() not in abc_services:
            continue
        if service_stat.get_abc_id() in ignore_abc_services:
            continue
        if cluster not in service_stat.clusters:
            continue
        cluster_stat = service_stat.clusters[cluster]
        volume_limit_map = ctx.obj.io_limit_map.get(service_stat.deploy_engine, {}).get(service_stat.service_id, {})
        for pod_id, pod_stat in cluster_stat.pods.items():
            if not pod_stat.has_storage_class(storage_class):
                continue

            pod_descriptor = PodDescriptor()
            pod_descriptor.service_id = service_stat.service_id
            pod_descriptor.pod_id = pod_id
            pod_descriptor.account_id = service_stat.account_id
            pod_descriptor.deploy_engine = service_stat.deploy_engine

            node_descriptor = ctx.obj.yp_stat.node_map.get(pod_stat.node_id)
            if node_descriptor is not None:
                pod_descriptor.rack = node_descriptor.rack

            pod_has_no_limit = False
            if service_stat.deploy_engine not in deploy_engines:
                pods_from_unknown_deploy_engine += 1
            elif service_stat.deploy_engine == "YP_LITE":
                for spec in pod_stat.iter_volume_desc(storage_class):
                    volume_descriptor = VolumeDescriptor()
                    volume_descriptor.mount_path = spec.mount_path
                    limit_map = volume_limit_map.get((spec.mount_path, storage_class))
                    if not spec.bandwidth_guarantee or not spec.bandwidth_limit or \
                        (not check_volume_stats(service_stat.service_id, service_stat.deploy_engine,
                                                ctx.obj.io_limit_map, spec) and ignore_pod_spec is True):
                        if limit_map:
                            volume_descriptor.bandwidth_guarantee = limit_map["guarantee"]
                            volume_descriptor.bandwidth_limit = limit_map["limit"]
                            volume_descriptor.storage_class = limit_map["storage_class"]
                            pod_descriptor.volumes.append(volume_descriptor)
                        else:
                            logging.warning("Pod %s/%s has no limits for volume %s", service_stat.service_id, pod_id,
                                            spec.mount_path)
                            pod_has_no_limit = True
            elif service_stat.deploy_engine == "QYP":
                storage_limit = volume_limit_map.get(storage_class)
                if storage_limit:
                    root_fs_bandwidth, one_volume_bandwidth = compute_qyp_bandwidth(storage_limit["guarantee"], pod_stat.iter_volume_desc(storage_class))

                    for spec in pod_stat.iter_volume_desc(storage_class):
                        volume_descriptor = VolumeDescriptor()
                        volume_descriptor.mount_path = spec.mount_path
                        bandwidth = root_fs_bandwidth if spec.mount_path == "/" else one_volume_bandwidth
                        if not bandwidth:
                            logging.warning("QYP VM %s has no limits for volume %s", service_stat.service_id, spec.mount_path)
                            pod_has_no_limit = True
                        elif not spec.bandwidth_guarantee or not spec.bandwidth_limit or ignore_pod_spec is True:
                            volume_descriptor.bandwidth_guarantee = bandwidth
                            volume_descriptor.storage_class = storage_limit["storage_class"]

                            pod_descriptor.volumes.append(volume_descriptor)

                    vols_with_guarantee = [i for i in pod_descriptor.volumes if i.bandwidth_guarantee > 0]
                    equal_limit = int(math.ceil(storage_limit["limit"] / max(len(vols_with_guarantee), 1)))

                    for vol in pod_descriptor.volumes:
                        if vol.bandwidth_guarantee > 0:
                            vol.bandwidth_limit = max(equal_limit, int(vol.bandwidth_guarantee))

                else:
                    logging.warning("QYP VM %s has no limits at all", service_stat.service_id)
                    pod_has_no_limit = True
            else:
                pods_from_unknown_deploy_engine += 1

            if pod_has_no_limit:
                pods_without_limits += 1

            pods_scanned += 1
            if pod_descriptor.volumes:
                pods.append(pod_descriptor)

    logging.info("Pods found: %d, pods without limits: %d, pods from unknown deploy engine: %d, pods scanned: %d",
                 len(pods), pods_without_limits, pods_from_unknown_deploy_engine, pods_scanned)

    safe_dump({
        "pods": [pod.to_dict() for pod in pods],
        "cluster": cluster
    }, sys.stdout, default_flow_style=False)
