# coding: utf-8

from __future__ import absolute_import, print_function

import sys
import logging
from collections import defaultdict

import click
from yaml import safe_dump

from infra.rtc.iolimit_ticketer.cli import cli
import infra.rtc.iolimit_ticketer.utils as utils


def compare_quota(quota_map, account_map, service_map, storage_class, deploy_engines):
    """
    :type quota_map: dict[int, dict[str, int]]
    :type account_map: dict[str, yp_model.AccountDescriptor]
    :type service_map: dict[(str, str), yp_model.ServiceDescriptor]
    """
    result = defaultdict(dict)
    grouped_services = utils.group_services_by_abc(
        service_map,
        target_storage_class=storage_class,
        target_deploy_engines=deploy_engines
    )
    for account in account_map.values():
        local_services = grouped_services.get(account.get_abc_id())
        if not local_services or utils.have_all_services_io_limits(local_services, storage_class):
            continue

        per_cluster_quotas = quota_map.get(account.get_abc_id())

        clusters = set()
        for service_stat in local_services.values():
            clusters.update(service_stat.clusters)

        for segments in account.clusters.values():
            if "default" not in segments:
                continue
            resources = segments["default"]

            if resources.cluster_name not in clusters:
                continue

            if storage_class not in resources.disks:
                continue
            disk_resources = resources.disks[storage_class]

            if per_cluster_quotas is None or resources.cluster_name not in per_cluster_quotas:
                result[account.get_abc_id()][resources.cluster_name] = None
                logging.warning("Account %s has no quota in %s", account.account_id, resources.cluster_name)
                continue

            multiplier = 1024 * 1024
            computed_quota = per_cluster_quotas.get(resources.cluster_name, {}).get(storage_class, 0) * multiplier
            if disk_resources.bandwidth_limits < computed_quota:
                result[account.get_abc_id()][resources.cluster_name] = (computed_quota - disk_resources.bandwidth_limits) / multiplier
                logging.warning("Account %s in %s has less quota than needed, %d < %d",
                                account.account_id, resources.cluster_name,
                                disk_resources.bandwidth_limits / multiplier, computed_quota / multiplier)

    safe_dump({"quota": dict(result)}, sys.stdout, default_flow_style=False)


@cli.command()
@click.option('--storage-class', default="ssd")
@click.option('--deploy-engines', default="")
@click.pass_context
def quota(ctx, storage_class, deploy_engines):
    """
    Compare quota.
    """
    deploy_engines = {x.strip() for x in deploy_engines.split(",") if x} or {"YP_LITE"}
    logging.info("deploy engines: %r, storage class: %r", deploy_engines, storage_class)
    compare_quota(ctx.obj.quota_map, ctx.obj.yp_stat.account_map, ctx.obj.yp_stat.service_map, storage_class, deploy_engines)
