# coding: utf-8

from __future__ import absolute_import, print_function

import difflib
import logging
from collections import defaultdict

import click
import infra.rtc.iolimit_ticketer.ok_client as ok_client

from infra.rtc.iolimit_ticketer.cli import cli
import infra.rtc.iolimit_ticketer.st_client as st_client
import infra.rtc.iolimit_ticketer.utils as utils


NEWLINE = u"\n"
STORAGE_CLASSES = ["ssd", "hdd"]


def get_vm_active_storages(vm_service):

    clusters = vm_service.clusters
    has_hdd, has_ssd = None, None

    for cluster in clusters.values():
        pods = cluster.pods

        for pod in pods.values():
            has_ssd, has_hdd = pod.has_storage_class("ssd"), pod.has_storage_class("hdd")

    return {"ssd": has_ssd, "hdd": has_hdd}


def update_description(issue, abc_service_id, abc_members, abc_services, yp_services, io_limit_map,
                       target_deploy_engines, dry_run=True):
    abc_name, abc_slug = abc_services[abc_service_id]

    description = []
    description.append(utils.render_template_from_resource(
        "/header_qyp.txt",
        abc_slug=abc_slug,
        abc_name=abc_name.get("en") or abc_name.get("ru")
    ))
    description.append(NEWLINE)
    description.append(utils.render_template_from_resource(
        '/tldr_qyp.txt'
    ))
    description.append(NEWLINE)
    description.append(NEWLINE)

    yp_services_map = defaultdict(list)
    for (deploy_engine, service_id), service_stat in yp_services.items():
        assert deploy_engine == "QYP"
        has_io_limit = service_stat.has_guarantee_and_limit("hdd") and service_stat.has_guarantee_and_limit("ssd")
        yp_services_map[has_io_limit].append(service_stat)

    has_not_equality = False

    service_counter = 0
    service_without_limit_counter = 0

    for key, service_list in sorted(yp_services_map.items()):
        has_io_limit, deploy_engine = key, "QYP"

        template_service_list = []
        for service_stat in sorted(service_list, key=lambda x: x.service_id):  # type: yp_model.ServiceDescriptor
            service_counter += 1

            active_storages = get_vm_active_storages(service_stat)

            io_guarantee_ssd = io_limit_map.get(deploy_engine, {}).get(service_stat.service_id,
                                                                       {}).get("ssd", {}).get("guarantee")

            io_guarantee_hdd = io_limit_map.get(deploy_engine, {}).get(service_stat.service_id,
                                                                       {}).get("hdd", {}).get("guarantee")

            if not io_guarantee_hdd and not io_guarantee_ssd:
                service_without_limit_counter += 1
            clusters = service_stat.clusters.keys()

            for dc in clusters:

                template_service = {
                    "cluster": dc,
                    "vm_id": service_stat.service_id
                }

                if active_storages["ssd"]:
                    template_service["io_guarantee_ssd"] = io_guarantee_ssd
                    template_service["has_guarantee_and_limit_ssd"] = service_stat.has_guarantee_and_limit("ssd")

                if active_storages["hdd"]:
                    template_service["io_guarantee_hdd"] = io_guarantee_hdd
                    template_service["has_guarantee_and_limit_hdd"] = service_stat.has_guarantee_and_limit("hdd")

                template_service_list.append(template_service)

        description.append(utils.render_template_from_resource(
            '/service_list_qyp.txt',
            deploy_engine=deploy_engine,
            has_io_limit=has_io_limit,
            service_list=template_service_list,
            storage_class="ssd"
        ))
        description.append(NEWLINE)

    description.append(utils.render_template_from_resource(
        '/footnotes.txt',
        has_not_equality=has_not_equality,
        service_without_limit_counter=service_without_limit_counter
    ))
    description.append(NEWLINE)

    description.append(utils.render_template_from_resource(
        '/instructions.txt',
        deploy_engines=target_deploy_engines

    ))
    description.append(NEWLINE)
    description.append(NEWLINE)
    description.append(utils.render_template_from_resource(
        '/footer_qyp.txt'
    ))
    description.append(NEWLINE)
    description.append(NEWLINE)

    description = u"".join(description)
    if issue.description != description:
        if dry_run:
            textdiff = u"\n".join(difflib.unified_diff(issue.description.splitlines(), description.splitlines(), n=1))
            logging.warning("Ticket %r for abc service %d should be updated:\n %s\n", issue, abc_service_id, textdiff)
        else:
            issue.update(description=description)

    logging.info("ABC %d, services found %d, services without modelled limit %d", abc_service_id, service_counter,
                 service_without_limit_counter)


def update_st_tickets_qyp(service_map, abc_members, abc_services, abc_responsibles, io_limit_map, dry_run,
                          target_abc_services, ignore_abc_services, target_deploy_engines, use_approvement,
                          ignore_activity):
    """
    :type service_map: dict[(str, str), yp_model.ServiceDescriptor]
    """
    client = st_client.create_st_client()
    ticket_map = st_client.get_ticket_map(client, "qyp")
    grouped_services = utils.group_services_by_abc(service_map, target_deploy_engines=['QYP'])

    for abc_service_id, local_services in grouped_services.items():

        if target_abc_services and abc_service_id not in target_abc_services:
            continue
        if abc_service_id in ignore_abc_services:
            continue

        if utils.have_all_services_io_limits(local_services, "ssd") and utils.have_all_services_io_limits(local_services, "hdd") and abc_service_id not in ticket_map:
            # services has limits, don't create new ticket for them
            continue

        if abc_service_id not in abc_services:
            logging.warning("Abc service %d is incomplete", abc_service_id)
            continue

        persons = {login for _, login in abc_members.get(abc_service_id, [])}
        persons.update(login for _, login in abc_responsibles.get(abc_service_id, []))
        persons = sorted(persons)
        if not persons:
            logging.warning("No persons found for abc service %d", abc_service_id)
            continue

        issue_key = st_client.get_or_create_ticket(client, ticket_map, abc_service_id, abc_services, "qyp",
                                                   persons, dry_run=dry_run)
        if issue_key is None:
            logging.warning("Ticket for abc service %d should be created", abc_service_id)
            continue

        issue = client.issues[issue_key]

        update_description(
            issue, abc_service_id, abc_members, abc_services, local_services, io_limit_map, target_deploy_engines,
            dry_run=dry_run
        )

        if utils.have_all_services_io_limits(local_services, "hdd") and \
           utils.have_all_services_io_limits(local_services, "ssd"):
            if dry_run:
                logging.warning("Ticket %r for abc service %d should be closed as ready", issue, abc_service_id)
            else:
                transition = issue.transitions['close']
                transition.execute(resolution='fixed', comment=u'Все лимиты применены, спасибо!')

        elif use_approvement and ok_client.get_approvement_id_from_issue(issue) is None:
            text = utils.render_template_from_resource('/approvement.txt')

            has_job = False

            for service_stat in local_services.values():

                active_storages = get_vm_active_storages(service_stat)

                if active_storages["ssd"]:
                    if not service_stat.has_guarantee_and_limit("ssd"):
                        has_job = True

                if active_storages["hdd"]:
                    if not service_stat.has_guarantee_and_limit("hdd"):
                        has_job = True

            if has_job and not st_client.has_activity(issue) or ignore_activity:
                if dry_run:
                    logging.warning("Approvement %r for abc service %d should be created", issue, abc_service_id)
                else:
                    assert persons
                    ok_client.post_approvement_to_issue(issue, text, persons)

    if not target_abc_services:
        logging.info("Closing unrelevant tickets")
        all_services = utils.group_services_by_abc(service_map, target_storage_class=None)
        st_client.close_unrelevant_tickets(client, ticket_map, all_services, dry_run=dry_run)
    else:
        logging.warning("Not closing unrelevant tickets")


@cli.command('qyp_tickets')
@click.option('--use-approvement/--no-approvement', default=False)
@click.option('--ignore-activity/--consider-activity', default=False)
@click.option('--abc-services', default="")
@click.option('--ignore-abc-services', default="")
@click.pass_context
def qyp_tickets(ctx, use_approvement, ignore_activity, abc_services, ignore_abc_services):
    """
    Generate tickets for QYP
    """
    abc_services = {int(x.strip()) for x in abc_services.split(",") if x}
    ignore_abc_services = {int(x.strip()) for x in ignore_abc_services.split(",") if x}
    ignore_abc_services.add(4172)
    logging.info("deploy engine: QYP, abc services: %r, ignore abc services: %r",
                 abc_services, ignore_abc_services)

    if not ctx.obj.dry_run:
        click.confirm('Do you want to continue?', abort=True)

    update_st_tickets_qyp(
        service_map=ctx.obj.yp_stat.service_map,
        abc_members=ctx.obj.abc_members,
        abc_services=ctx.obj.abc_services,
        abc_responsibles=ctx.obj.abc_responsibles,
        io_limit_map=ctx.obj.io_limit_map,
        dry_run=ctx.obj.dry_run,
        target_abc_services=abc_services,
        ignore_abc_services=ignore_abc_services,
        target_deploy_engines={'QYP'},
        use_approvement=use_approvement,
        ignore_activity=ignore_activity
    )
