# coding: utf-8

from __future__ import absolute_import, print_function

import difflib
import logging
from collections import defaultdict

import click

from infra.rtc.iolimit_ticketer.cli import cli
import infra.rtc.iolimit_ticketer.st_client as st_client
import infra.rtc.iolimit_ticketer.nanny_client as nanny_client
import infra.rtc.iolimit_ticketer.ok_client as ok_client
import infra.rtc.iolimit_ticketer.utils as utils

NEWLINE = u"\n"
DEFAULT_DEPLOY_ENGINES = {"YP_LITE", "MCRSC", "RSC"}
ALL_DEPLOY_ENGINES = {"YP_LITE", "MCRSC", "RSC"}
NANNY_NOTIFICATION_PREFIX_TEMPLATE = "io-{}-read-limit-"
BLACKLIST_ABC_ID = {31632, 30776}


def build_nanny_volume_cgi(volume_limit_map, mount_paths, storage_class):
    if not volume_limit_map:
        return True, ["rootIoGuarantee=0", "rootIoLimit=0"]
    result = []
    has_no_limit = True
    for mount_path in mount_paths:
        volume_limit_spec = volume_limit_map.get((mount_path, storage_class))
        if not volume_limit_spec or volume_limit_spec["storage_class"] != storage_class:
            continue
        has_no_limit = False
        if mount_path == "/":
            result.append("rootIoGuarantee={}".format(volume_limit_spec["guarantee"]))
            result.append("rootIoLimit={}".format(volume_limit_spec["limit"]))
        else:
            result.append("volumeIoGuarantee={},{}".format(mount_path, volume_limit_spec["guarantee"]))
            result.append("volumeIoLimit={},{}".format(mount_path, volume_limit_spec["limit"]))
    return has_no_limit, result


def build_yasm_url(pod_stat, cluster_stat):
    if pod_stat.yasm_tags:
        geo = cluster_stat.cluster_name
        if geo in ("iva", "myt"):
            geo = "msk"
        return "https://yasm.yandex-team.ru/template/panel/Porto-container/hosts=ASEARCH;itype={itype};ctype={ctype};prj={prj};geo={geo}".format(
            itype=pod_stat.yasm_tags["itype"],
            ctype=pod_stat.yasm_tags["ctype"],
            prj=pod_stat.yasm_tags["prj"],
            geo=geo
        )
    else:
        return None


def update_description(client, issue, abc_service_id, abc_members, abc_services, yp_services, io_limit_map,
                       storage_class, nanny_modifier, target_deploy_engines, dry_run=True):
    abc_name, abc_slug = abc_services[abc_service_id]

    description = []
    description.append(utils.render_template_from_resource(
        "/header.txt",
        abc_slug=abc_slug,
        abc_name=abc_name.get("en") or abc_name.get("ru"),
        storage_class=storage_class
    ))
    description.append(NEWLINE)
    description.append(utils.render_template_from_resource(
        '/tldr.txt',
        storage_class=storage_class
    ))
    description.append(NEWLINE)
    description.append(NEWLINE)

    yp_services_map = defaultdict(list)
    for (deploy_engine, service_id), service_stat in yp_services.items():
        has_io_limit = service_stat.has_guarantee_and_limit(storage_class)
        yp_services_map[(deploy_engine, has_io_limit)].append(service_stat)

    has_not_equality = False

    service_counter = 0
    service_without_limit_counter = 0

    for key, service_list in sorted(yp_services_map.items()):
        deploy_engine, has_io_limit = key

        template_service_list = []
        for service_stat in sorted(service_list, key=lambda x: x.service_id):  # type: yp_model.ServiceDescriptor
            service_counter += 1
            if deploy_engine == "YP_LITE":
                io_guarantee = 0
                io_limit = 0
                volume_limit_map = io_limit_map.get(deploy_engine, {}).get(service_stat.service_id, {})
                for volume_limit_spec in volume_limit_map.values():
                    if not volume_limit_spec or not volume_limit_spec.get("storage_class")\
                            or volume_limit_spec["storage_class"] != storage_class:
                        continue
                    io_guarantee += volume_limit_spec["guarantee"]
                    io_limit += volume_limit_spec["guarantee"]

                has_no_limit_overall = False
                template_cluster_list = []
                for cluster_stat in sorted(service_stat.clusters.values(), key=lambda x: x.cluster_name):
                    first_pod = cluster_stat.get_first_pod_id(storage_class)
                    mount_paths = cluster_stat.get_mount_paths(storage_class)
                    if not first_pod or not mount_paths:
                        continue

                    has_no_limit, volume_cgi = build_nanny_volume_cgi(volume_limit_map, mount_paths, storage_class)
                    has_no_limit_overall = has_no_limit_overall or has_no_limit

                    has_different_pods = len({pod_stat.resource_desc.to_tuple() for pod_stat in cluster_stat.pods.values()}) != 1
                    if has_different_pods:
                        has_not_equality = True

                    template_cluster_list.append({
                        "has_guarantee_and_limit": cluster_stat.has_guarantee_and_limit(storage_class),
                        "has_different_pods": has_different_pods,
                        "first_pod": first_pod,
                        "cluster_name": cluster_stat.cluster_name,
                        "volume_cgi": "&".join(volume_cgi),
                        "yasm_url": build_yasm_url(cluster_stat.pods[first_pod], cluster_stat)
                    })

                if has_no_limit_overall:
                    service_without_limit_counter += 1

                template_service = {
                    "service_id": service_stat.service_id,
                    "clusters": template_cluster_list,
                    "has_no_limit_overall": has_no_limit_overall,
                    "io_guarantee": io_guarantee,
                    "io_limit": io_limit
                }

                if not has_io_limit:
                    nanny_modifier.update_notification(
                        issue_key=issue.key,
                        **template_service
                    )

            elif deploy_engine in ("MCRSC", "RSC"):
                io_guarantee = io_limit_map.get(deploy_engine, {}).get(service_stat.service_id, {}).get(storage_class, {}).get("guarantee")
                if not io_guarantee:
                    service_without_limit_counter += 1
                template_service = {
                    "deploy_unit_id": service_stat.deploy_unit_id,
                    "stage_id": service_stat.deploy_stage_id,
                    "has_guarantee_and_limit": service_stat.has_guarantee_and_limit(storage_class),
                    "io_guarantee": io_guarantee
                }

            else:
                raise Exception("{} not supported".format(deploy_engine))

            template_service_list.append(template_service)

        description.append(utils.render_template_from_resource(
            '/service_list_{}.txt'.format(deploy_engine.lower()),
            deploy_engine=deploy_engine,
            has_io_limit=has_io_limit,
            service_list=template_service_list,
            storage_class=storage_class
        ))
        description.append(NEWLINE)

    description.append(utils.render_template_from_resource(
        '/footnotes.txt',
        has_not_equality=has_not_equality,
        service_without_limit_counter=service_without_limit_counter,
        storage_class=storage_class
    ))
    description.append(NEWLINE)

    description.append(utils.render_template_from_resource(
        '/instructions.txt',
        storage_class=storage_class,
        deploy_engines=target_deploy_engines
    ))
    description.append(NEWLINE)
    description.append(NEWLINE)
    description.append(utils.render_template_from_resource(
        '/footer.txt',
        storage_class=storage_class
    ))
    description.append(NEWLINE)
    description.append(NEWLINE)

    description = u"".join(description)
    if issue.description != description:
        if dry_run:
            textdiff = u"\n".join(difflib.unified_diff(issue.description.splitlines(), description.splitlines(), n=1))
            logging.warning("Ticket %r for abc service %d should be updated:\n %s\n", issue, abc_service_id, textdiff)
        else:
            issue.update(description=description)

    logging.info("ABC %d, services found %d, services without modelled limit %d", abc_service_id, service_counter, service_without_limit_counter)


def update_st_tickets(service_map, abc_members, abc_services, abc_responsibles, io_limit_map,
                      dry_run, use_approvement, storage_class, target_deploy_engines,
                      target_abc_services, ignore_abc_services, ignore_activity):
    """
    :type service_map: dict[(str, str), yp_model.ServiceDescriptor]
    """
    client = st_client.create_st_client()
    ticket_map = st_client.get_ticket_map(client, storage_class)

    nanny_modifier = nanny_client.NannyServiceModifier(
        template_name="/nanny_{}.txt".format(storage_class),
        prefix=NANNY_NOTIFICATION_PREFIX_TEMPLATE.format(storage_class),
        dry_run=dry_run
    )

    grouped_services = utils.group_services_by_abc(service_map, target_storage_class=storage_class, target_deploy_engines=target_deploy_engines)
    for abc_service_id, local_services in grouped_services.items():
        if target_abc_services and abc_service_id not in target_abc_services:
            continue
        if abc_service_id in ignore_abc_services:
            continue

        if utils.have_all_services_io_limits(local_services, storage_class) and abc_service_id not in ticket_map:
            # services has limits, don't create new ticket for them
            continue

        if abc_service_id not in abc_services or abc_service_id not in abc_members:
            logging.warning("Abc service %d is incomplete", abc_service_id)
            continue

        persons = {login for _, login in abc_members.get(abc_service_id, [])}
        persons.update(login for _, login in abc_responsibles.get(abc_service_id, []))
        persons = sorted(persons)
        if not persons:
            logging.warning("No persons found for abc service %d", abc_service_id)
            continue

        issue_key = st_client.get_or_create_ticket(client, ticket_map, abc_service_id, abc_services, storage_class, persons, dry_run=dry_run)
        if issue_key is None:
            logging.warning("Ticket for abc service %d should be created", abc_service_id)
            continue

        issue = client.issues[issue_key]
        update_description(
            client=client,
            issue=issue,
            abc_service_id=abc_service_id,
            abc_members=abc_members,
            abc_services=abc_services,
            yp_services=local_services,
            io_limit_map=io_limit_map,
            storage_class=storage_class,
            nanny_modifier=nanny_modifier,
            target_deploy_engines=target_deploy_engines,
            dry_run=dry_run
        )

        if utils.have_all_services_io_limits(local_services, storage_class):
            if dry_run:
                logging.warning("Ticket %r for abc service %d should be closed as ready", issue, abc_service_id)
            else:
                transition = issue.transitions['close']
                transition.execute(resolution='fixed', comment=u'Все лимиты применены, спасибо!')

        elif use_approvement and ok_client.get_approvement_id_from_issue(issue) is None:
            text = utils.render_template_from_resource('/approvement.txt', storage_class=storage_class)

            has_job = False
            for service_stat in local_services.values():
                if service_stat.deploy_engine == "YP_LITE" and not service_stat.has_guarantee_and_limit(storage_class):
                    has_job = True

            if has_job and (not st_client.has_activity(issue) or ignore_activity):
                if dry_run:
                    logging.warning("Approvement %r for abc service %d should be created", issue, abc_service_id)
                else:
                    assert persons
                    ok_client.post_approvement_to_issue(issue, text, persons)

    if not target_abc_services and not ignore_abc_services:
        logging.info("Closing unrelevant tickets")
        all_services = utils.group_services_by_abc(service_map, target_storage_class=storage_class)
        st_client.close_unrelevant_tickets(client, ticket_map, all_services, dry_run=dry_run)
    else:
        logging.warning("Not closing unrelevant tickets")

    if not target_abc_services and ignore_abc_services == BLACKLIST_ABC_ID and "YP_LITE" in target_deploy_engines:
        logging.info("Cleanup nanny notifications")
        nanny_modifier.cleanup()
    else:
        logging.warning("Not updating nanny notifications")


@cli.command()
@click.option('--use-approvement/--no-approvement', default=False)
@click.option('--ignore-activity/--consider-activity', default=False)
@click.option('--storage-class', default="ssd")
@click.option('--deploy-engines', default="")
@click.option('--abc-services', default="")
@click.option('--ignore-abc-services', default="")
@click.pass_context
def tickets(ctx, use_approvement, ignore_activity, storage_class, deploy_engines, abc_services, ignore_abc_services):
    """
    Generate tickets.
    """
    deploy_engines = {x.strip() for x in deploy_engines.split(",") if x} or DEFAULT_DEPLOY_ENGINES
    abc_services = {int(x.strip()) for x in abc_services.split(",") if x}
    ignore_abc_services = {int(x.strip()) for x in ignore_abc_services.split(",") if x}

    for abc_id in list(BLACKLIST_ABC_ID):
        ignore_abc_services.add(abc_id)

    logging.info("use approvement: %r, ignore activity: %r, storage class: %r, deploy engines: %r, abc services: %r, ignore abc services: %r",
                 use_approvement, ignore_activity, storage_class, deploy_engines, abc_services, ignore_abc_services)
    assert ALL_DEPLOY_ENGINES.issuperset(deploy_engines), "wrong deploy engines given"

    if not ctx.obj.dry_run:
        click.confirm('Do you want to continue?', abort=True)

    update_st_tickets(
        service_map=ctx.obj.yp_stat.service_map,
        abc_members=ctx.obj.abc_members,
        abc_services=ctx.obj.abc_services,
        abc_responsibles=ctx.obj.abc_responsibles,
        io_limit_map=ctx.obj.io_limit_map,
        dry_run=ctx.obj.dry_run,
        use_approvement=use_approvement,
        storage_class=storage_class,
        target_deploy_engines=deploy_engines,
        target_abc_services=abc_services,
        ignore_abc_services=ignore_abc_services,
        ignore_activity=ignore_activity
    )
