#!/home/dldmitry/walle_venv/bin/python
from __future__ import print_function

import os
import json
from collections import defaultdict, Counter

USE_INFRA_CACHE = True

datacenter = "vla"

average_cache_hit = defaultdict(lambda: {"sum": 0, "count": 0})
cache_hit_list = defaultdict(list)

per_pod_cache_hit = defaultdict(lambda: {"sum": 0, "count": 0})
per_pod_hit_list = defaultdict(list)
infra_cache_list = defaultdict(list)

per_host_cache_size = defaultdict(int)
per_pod_cache_size = defaultdict(int)
infra_cache_size = defaultdict(int)

yp_hosts = set(json.load(open("yp_hosts.json")))

# "PORTO_LAYER_SEARCH_UBUNTU_PRECISE_APP", "PORTO_LAYER_SEARCH_UBUNTU_TRUSTY_APP", "PORTO_LAYER_SEARCH_UBUNTU_XENIAL_APP",
# "PORTO_LAYER_SEARCH_UBUNTU_BIONIC_APP", "PORTO_LAYER_SEARCH_UBUNTU_TRUSTY", "PORTO_LAYER_SEARCH_UBUNTU_PRECISE", "PORTO_LAYER_SEARCH_UBUNTU_XENIAL", "PORTO_LAYER_SEARCH_UBUNTU_BIONIC",
# "PORTO_LAYER_SEARCH_UBUNTU_TRUSTY_SUBAGENT", "PORTO_LAYER_SEARCH_UBUNTU_PRECISE_SUBAGENT", "PORTO_LAYER_SEARCH_UBUNTU_XENIAL_SUBAGENT", "PORTO_LAYER_SEARCH_UBUNTU_BIONIC_SUBAGENT",
# "PORTO_LAYER_BALANCER_MTN"
base_images = set(x.strip("rbtorrent:") for x in json.load(open("base_images.json")))
infra_resource_names = {
    "balancer",
    "gdb",
    "gdb.squashfs",
    "gdb.tar",
    "gdb.tar.gz",
    "gdb.tgz",
    "gdb_toolkit.tgz",
    "instancectl",
    "iss_hook_install",
    "iss_hook_notify",
    "iss_hook_reopenlogs",
    "iss_hook_start",
    "iss_hook_status",
    "iss_hook_stop",
    "iss_hook_uninstall",
    "juggler-check-bundle-rtc-balancers.tar.gz",
    "juggler-scheduler",
    "logrotate",
    "push-client",
    "push_client",
    "pushclient",
    "tvmtool",
    "vmagent",
    "vmctl",
    "solomon_agent",
    "solomon-agent",
    "tvm-tool",
    "unified_agent",
    "vmproxy",
    "breakpad_init.so",
    "dump_json_get_workers_provider.lua"
}


def fix_storage_class(storage_class):
    if storage_class == "nvme":
        return "ssd"
    else:
        return storage_class


def process_spec(spec):
    hostname = spec["hostname"]
    capacities = spec["capacities"]
    resources = spec["resources"]
    if not resources or not capacities:
        return

    container_use_our_base_layer = set()
    for resource in resources:
        resource_uuid = resource["uuid"]
        if not resource_uuid or resource["type"] != "layer" or resource["container_fqdn"] == hostname:
            continue
        if resource_uuid.strip("layer_rbtorrent_") in base_images:
            container_use_our_base_layer.add(resource["container_fqdn"])

    storage_class_set = {fix_storage_class(capacity_spec["storage_class"]) for capacity_spec in capacities}
    total_resource_size = defaultdict(int)
    per_pod_resource_cache = defaultdict(lambda: defaultdict(lambda: defaultdict(list)))
    for resource in resources:
        storage_class = fix_storage_class(resource["storage_class"])
        resource_capacity = resource["resource_capacity"]
        if not resource_capacity:
            continue
        total_resource_size[storage_class] += resource_capacity

        resource_uuid = resource["uuid"]
        if not resource_uuid:
            continue

        resource_name = resource.get("resource_name")
        container_fqdn = resource["container_fqdn"] or hostname
        has_shared_resource = bool(resource_name and resource_name in infra_resource_names)
        has_shared_layer = bool(resource["type"] in ("layer", "unpacked_layer") and container_fqdn in container_use_our_base_layer)
        if USE_INFRA_CACHE and (has_shared_layer or has_shared_resource):
            container_fqdn = hostname
            storage_class = "hdd" if "hdd" in storage_class_set else "ssd"
        per_pod_resource_cache[storage_class][container_fqdn][resource_uuid].append(resource)

    for capacity_spec in capacities:
        storage_class = fix_storage_class(capacity_spec["storage_class"])
        total_storage_capacity = capacity_spec["total"]
        total_resource_capacity = total_resource_size[storage_class]
        if not total_storage_capacity or not total_resource_capacity:
            continue

        # average cache hit
        hit = total_resource_capacity * 1.0 / total_storage_capacity
        if hit:
            cache_hit_list[storage_class].append(hit)
            x = average_cache_hit[storage_class]
            x["sum"] += hit
            x["count"] += 1

        # per pod resource cache
        per_pod_cache_capacity = 0
        for fqdn, pod_resource_cache in per_pod_resource_cache[storage_class].items():
            pod_capacity = 0
            for pod_resource_list in pod_resource_cache.values():
                assert len({x["resource_capacity"] for x in pod_resource_list}) == 1
                pod_capacity += pod_resource_list[0]["resource_capacity"]
            per_pod_cache_capacity += pod_capacity
            if fqdn == hostname:
                if pod_capacity > 30 * 1024 * 1024 * 1024:
                    print(fqdn)
                    print(json.dumps(sorted({y["service_id"] for x in per_pod_resource_cache[storage_class][fqdn].values() for y in x if not y["container_fqdn"]}), indent=4))
                infra_cache_size[storage_class] += pod_capacity
                infra_cache_list[storage_class].append(pod_capacity)

        per_pod_hit = per_pod_cache_capacity * 1.0 / total_storage_capacity
        if per_pod_hit:
            per_pod_hit_list[storage_class].append(per_pod_hit)
            x = per_pod_cache_hit[storage_class]
            x["sum"] += per_pod_hit
            x["count"] += 1

        # per host resource cache capacity
        per_host_cache_size[storage_class] += total_storage_capacity

        # per pod resource cache capacity
        per_pod_cache_size[storage_class] += per_pod_cache_capacity


def compute_average_cache_hit():
    for storage_class, x in average_cache_hit.items():
        print("average no cache / per host cache rate (size):", storage_class, x["sum"] / x["count"])
    for storage_class, l in cache_hit_list.items():
        l.sort()
        print("min, median, 95p, max no cache / per host rate (size):", storage_class, l[0], l[len(l)/2], l[int(len(l)*0.95)], l[-1])


def compute_per_pod_cache_hit():
    for storage_class, x in per_pod_cache_hit.items():
        print("average per pod cache / per host cache rate (size):", storage_class, x["sum"] / x["count"])
    for storage_class, l in per_pod_hit_list.items():
        l.sort()
        print("min, median, 95p, max per pod cache / per host cache rate (size):", storage_class, l[0], l[len(l)/2], l[int(len(l)*0.95)], l[-1])
    for storage_class, l in infra_cache_list.items():
        l.sort()
        print("min, median, 95p, max per infra cache, MB:", storage_class, l[0] / 1024. / 1024., l[len(l)/2] / 1024. / 1024., l[int(len(l)*0.95)] / 1024. / 1024., l[-1] / 1024. / 1024.)


def compute_resource_cache_capacity():
    for storage_class, x in per_host_cache_size.items():
        print("per host cache total size, TB:", storage_class, x / 1024. / 1024. / 1024. / 1024.)
    for storage_class, x in per_pod_cache_size.items():
        print("per pod cache total size, TB:", storage_class, (x - infra_cache_size.get(storage_class, 0)) / 1024. / 1024. / 1024. / 1024.)
    for storage_class, x in infra_cache_size.items():
        print("infra cache total size, TB:", storage_class, x / 1024. / 1024. / 1024. / 1024.)


def main():
    root_path = "data"
    for name in os.listdir(root_path):
        if datacenter not in name:
            continue
        if name[:-5] not in yp_hosts:
            continue
        spec = json.load(open("{}/{}".format(root_path, name)))
        process_spec(spec)
    compute_average_cache_hit()
    compute_per_pod_cache_hit()
    compute_resource_cache_capacity()


if __name__ == "__main__":
    main()

