#!/skynet/python/bin/python
# coding: utf-8
from __future__ import print_function

import os
import sys
import json
import requests
import subprocess
import socket
import time
import random

from api.cqueue import Client
from library.sky.hostresolver import Resolver

SPACE_CACHE = {}


def sleep_random(ceil):
    time.sleep(random.random() * ceil)
    return


def find_layers(meta_container):
    mount_point = None
    try:
        porto_output = subprocess.check_output(["portoctl", "vlist", "-c", meta_container, "-v"])
    except subprocess.CalledProcessError:
        return []
    for line in porto_output.splitlines():
        if line.startswith("  "):
            parts = line.strip().split(" ")
            key = parts[0]
            value = parts[-1]
            if key == "layers" and mount_point == "/":
                return ["/place/porto_layers/{}".format(x) for x in value.split(";")]
        elif line.startswith("/"):
            mount_point = line.strip()
    return []


def get_space(folder, sudo=False):
    if folder in SPACE_CACHE:
        return SPACE_CACHE[folder]
    sleep_random(10)
    sudo_args = []
    if sudo:
        sudo_args.append("sudo")
    try:
        du_output = subprocess.check_output(sudo_args + ["du", "--bytes", "--one-file-system", "--summarize", folder])
    except subprocess.CalledProcessError:
        return None
    else:
        space = SPACE_CACHE[folder] = int(du_output.split("\t")[0])
        return space


def get_places():
    with open("/etc/server_info.json") as stream:
        server_info = json.load(stream)
    for mount_point, spec in server_info["points_info"].items():
        if spec["mount_point"] == mount_point and mount_point in ["/place", "/ssd"]:
            disk_type = [y["disk_type"] for x in spec["disks_info"] for y in x.values()][0]
            yield mount_point, disk_type


def collect_info():
    resource_info = []
    capacity_info = []

    for mount_point, storage_class in get_places():
        resources_folder = "{}/db/iss3/resources".format(mount_point)
        shared_resources = os.listdir(resources_folder)
        instances = requests.get("http://localhost:25536/instances").json()
        for container_spec in instances:
            meta_container = container_spec["metaContainerPath"]
            deploy_engine = container_spec["instanceData"].get("properties/DEPLOY_ENGINE")
            service_id = container_spec["instanceData"].get("properties/NANNY_SERVICE_ID")
            container_fqdn = container_spec["instanceData"].get("properties/HOSTNAME")
            dump_path = os.path.join(container_spec["instanceDir"], "dump.json")
            with open(dump_path) as stream:
                dump_spec = json.load(stream)

            resources = dump_spec.get("resources")
            for resource_name, resource_spec in resources.items():
                resource_uuid = resource_spec.get("uuid")
                if resource_uuid is None:
                    continue
                resource_folder = None
                for folder_name in shared_resources:
                    if folder_name.startswith(resource_uuid):
                        resource_folder = os.path.join(resources_folder, folder_name)
                        break
                if resource_folder is None:
                    continue
                resource_capacity = get_space(resource_folder)
                resource_info.append({
                    "deploy_engine": deploy_engine,
                    "service_id": service_id,
                    "resource_name": resource_name,
                    "uuid": resource_uuid,
                    "resource_capacity": resource_capacity,
                    "container_fqdn": container_fqdn,
                    "storage_class": storage_class,
                    "type": "resource"
                })

            volumes = dump_spec.get("volumes", [])
            for volume_spec in volumes:
                layers = volume_spec.get("layers")
                if not layers:
                    continue
                for layer_spec in layers:
                    layer_uuid = layer_spec.get("uuid")
                    if layer_uuid is None:
                        continue
                    layer_folder = None
                    for folder_name in shared_resources:
                        if folder_name.startswith(layer_uuid):
                            layer_folder = os.path.join(resources_folder, folder_name)
                            break
                    if layer_folder is None:
                        continue
                    layer_capacity = get_space(layer_folder)
                    resource_info.append({
                        "deploy_engine": deploy_engine,
                        "service_id": service_id,
                        "uuid": layer_uuid,
                        "resource_capacity": layer_capacity,
                        "container_fqdn": container_fqdn,
                        "storage_class": storage_class,
                        "type": "layer"
                    })

            if mount_point == "/place":
                for layer_path in find_layers(meta_container):
                    layer_capacity = get_space(layer_path, sudo=True)
                    resource_info.append({
                        "deploy_engine": deploy_engine,
                        "service_id": service_id,
                        "uuid": layer_path,
                        "resource_capacity": layer_capacity,
                        "container_fqdn": container_fqdn,
                        "storage_class": storage_class,
                        "type": "unpacked_layer"
                    })

        total_capacity = get_space(resources_folder)
        capacity_info.append({
            "total": total_capacity,
            "storage_class": storage_class
        })

    return {
        "resources": resource_info,
        "capacities": capacity_info,
        "hostname": socket.getfqdn()
    }


def collect_info_wrapper(tries=5):
    sleep_random(600)
    for idx in xrange(tries):
        try:
            return collect_info()
        except (IOError, OSError):
            if tries - 1 == idx:
                raise
            else:
                sleep_random(10)
    raise Exception("logic error")


def get_path(host):
    return "data/{}.json".format(host)


def main():
    with Client() as client:
        hosts = json.loads(subprocess.check_output("./find_hosts.py"))
        hosts = [x for x in hosts if not os.path.exists(get_path(x))]
        hosts.sort()
        print("Hosts left: {}".format(len(hosts)))
        assert hosts

        with client.iter(hosts, collect_info_wrapper) as session:
            for host, result, err in session.wait():
                if isinstance(err, StopIteration):
                    continue
                if result is not None:
                    print("Получен результат от %r: %s" % (host, result["capacities"]))
                    with open(get_path(host), "w") as stream:
                        json.dump(result, stream)
                else:
                    print("Получен результат от %r: %s" % (host, err))


if __name__ == '__main__':
    main()
