# coding: utf-8

from __future__ import absolute_import, print_function

import logging
import time
from requests.exceptions import ReadTimeout

import nanny_rpc_client
from nanny_repo import repo_api_pb2, repo_api_stub, repo_pb2
from google.protobuf.field_mask_pb2 import FieldMask

import infra.rtc.iolimit_ticketer.utils as utils

NANNY_URL = 'https://nanny.yandex-team.ru/api/repo'


class NannyClient(object):

    def __init__(self, dry_run=True, severity_level=None):
        self._rpc_client = nanny_rpc_client.RetryingRpcClient(NANNY_URL, utils.get_nanny_oauth_token())
        self._repo_stub = repo_api_stub.RepoServiceStub(self._rpc_client)
        self._dry_run = dry_run
        self._severity_level = severity_level or repo_pb2.PerServiceNotification.DANGER

    def list_existing_notifications(self, prefix):
        field_mask = FieldMask(paths=["meta", "spec"])
        list_notifications_request = repo_api_pb2.ListNotificationsRequest(field_mask=field_mask)
        for notification in self._repo_stub.list_notifications(list_notifications_request).notifications:
            if notification.meta.id.startswith(prefix):
                yield notification

    def get_notification_map(self, prefix, map_func):
        result = {}
        for notification in self.list_existing_notifications(prefix):
            key = map_func(notification.meta.id[len(prefix):])
            result[key] = notification
        return result

    def remove_existing_notification(self, notification_id):
        remove_notification_request = repo_api_pb2.RemoveNotificationRequest(notification_id=notification_id)
        if self._dry_run:
            logging.warning("Nanny notification should be removed, %r", remove_notification_request)
        else:
            self._repo_stub.remove_notification(remove_notification_request)
            logging.warning("Removed nanny notifications %r", remove_notification_request)

    def update_nanny_notification(self, service_list, nanny_notification_id, content, existing_notification=None):
        if not service_list:
            if existing_notification is not None:
                self.remove_existing_notification(existing_notification.meta.id)
            return

        per_service = repo_pb2.PerServiceNotification(
            content=content,
            severity_level=self._severity_level,
            services=[repo_pb2.ServiceRef(id=service_name) for service_name in sorted(service_list)]
        )

        if existing_notification is None:
            meta = repo_pb2.NotificationMeta(
                id=nanny_notification_id
            )
            spec = repo_pb2.NotificationSpec(
                type=repo_pb2.NotificationSpec.PER_SERVICE,
                per_service=per_service
            )
            create_request = repo_api_pb2.CreateNotificationRequest(meta=meta, spec=spec)

            if self._dry_run:
                logging.warning("Nanny notification should be created, %r", create_request)
            else:
                self._repo_stub.create_notification(create_request)

            return

        services_with_notifications = {service.id for service in existing_notification.spec.per_service.services}
        if set(service_list) == services_with_notifications and existing_notification.spec.per_service.content == content:
            return

        existing_notification.spec.per_service.CopyFrom(per_service)
        update_request = repo_api_pb2.UpdateNotificationRequest(
            meta=existing_notification.meta,
            spec=existing_notification.spec
        )

        if self._dry_run:
            logging.warning("Nanny notification should be updated, %r", update_request)
        else:
            self._repo_stub.update_notification(update_request)


class NannyServiceModifier(object):

    def __init__(self, template_name, prefix, dry_run=True, severity_level=None):
        self._template_name = template_name
        self._prefix = prefix
        self._nanny_client = NannyClient(dry_run=dry_run, severity_level=severity_level)
        self._notification_map = self._nanny_client.get_notification_map(self._prefix, map_func=str)
        self._seen_ids = set()

    def update_notification(self, service_id, issue_key, **kwargs):
        nanny_notification_id = "{}{}".format(self._prefix, service_id)
        self._seen_ids.add(nanny_notification_id)

        content = utils.render_template_from_resource(self._template_name, service_id=service_id, issue_key=issue_key, **kwargs)
        self._nanny_client.update_nanny_notification(
            [service_id], nanny_notification_id, content, self._notification_map.get(service_id)
        )

    def remove_notification(self, service_id):
        if service_id in self._notification_map:
            notification = self._notification_map.pop(service_id)
            self._nanny_client.remove_existing_notification(notification.meta.id)

    def cleanup(self):
        for notification in self._notification_map.values():
            if notification.meta.id not in self._seen_ids:
                self._nanny_client.remove_existing_notification(notification.meta.id)

    # FIXME: костыль для удаления всех существующих плашек nanny
    def remove_all_notifications(self):
        for notification in self._notification_map.values():
            try:
                self._nanny_client.remove_existing_notification(notification.meta.id)
            except ReadTimeout:
                time.sleep(40)
                self._nanny_client.remove_existing_notification(notification.meta.id)
