# coding: utf-8

from __future__ import absolute_import, print_function

import re
import json
import uuid
from datetime import datetime, timedelta

import requests
from dateutil.parser import parse as dt_parse
from dateutil.tz import tzutc

import infra.rtc.iolimit_ticketer.utils as utils

TICKET_COMMENT = """{{{{iframe src="https://ok.yandex-team.ru/approvements/{}?_embedded=1" frameborder=0 width=100% height=400px scrolling=no}}}}"""
EXPIRE_DAYS = 14


class Approvement(object):

    def __init__(self):
        self.uuid = None
        self.status = None
        self.resolution = None
        self.created = None

    def get_ticket_comment(self):
        assert self.uuid is not None
        return TICKET_COMMENT.format(self.uuid)

    def is_expired(self):
        assert self.created is not None and self.status is not None
        if self.status == "in_progress":
            return datetime.now(tz=tzutc()) - self.created > timedelta(days=EXPIRE_DAYS)
        else:
            return False

    def is_approved(self):
        assert self.resolution is not None and self.status is not None
        return self.resolution == "approved" and self.status == "closed"

    def is_rejected(self):
        assert self.status is not None
        return self.status == "rejected"

    def is_declined(self):
        assert self.resolution is not None and self.status is not None
        return self.resolution == "declined" and self.status == "closed"

    @classmethod
    def from_dict(cls, reply):
        approvement = cls()
        approvement.uuid = reply["uuid"]
        approvement.status = reply["status"]
        approvement.resolution = reply["resolution"]
        approvement.created = dt_parse(reply["created"])
        return approvement


def _create_url(path):
    return "https://ok.yandex-team.ru{}".format(path)


def _create_headers():
    return {
        'Authorization': 'OAuth {}'.format(utils.get_ok_oauth_token()),
        'Content-Type': 'application/json'
    }


def _make_post_request(path, payload):
    r = requests.post(
        _create_url(path), headers=_create_headers(), data=json.dumps(payload)
    )
    r.raise_for_status()
    return r.json()


def _make_get_request(path):
    r = requests.get(_create_url(path), headers=_create_headers())
    r.raise_for_status()
    return r.json()


def create_approvement(ticket_id, text, approvers):
    payload = {
        'is_parallel': True,
        'is_reject_allowed': True,
        'object_id': ticket_id,
        'groups': ['svc_gencfg'],
        'stages': [{'need_all': False, 'stages': [{'approver': approver} for approver in approvers]}],
        'text': text,
        'uid': uuid.uuid4().hex
    }
    reply = _make_post_request("/api/approvements/", payload)
    return Approvement.from_dict(reply)


def get_approvement(approvement_id):
    reply = _make_get_request("/api/approvements/{}/".format(approvement_id))
    return Approvement.from_dict(reply)


def close_approvement(approvement_id):
    reply = _make_post_request("/api/approvements/{}/close/".format(approvement_id), {})
    return Approvement.from_dict(reply)


def get_approvement_id_from_issue(issue):
    matcher = re.compile(TICKET_COMMENT.format("(.+)"))
    approvement_id = None
    for comment in issue.comments:
        m = matcher.search(comment.text)
        if m is not None:
            approvement_id = m.group(1)
    return approvement_id


def get_approvement_from_issue(issue):
    approvement_id = get_approvement_id_from_issue(issue)
    if approvement_id is not None:
        return get_approvement(approvement_id)
    return None


def post_approvement_to_issue(issue, text, approvers):
    assert get_approvement_id_from_issue(issue) is None
    approvement = create_approvement(issue.key, text, approvers)
    issue.comments.create(text=approvement.get_ticket_comment())
