# coding: utf-8

from __future__ import absolute_import, print_function

import time
import logging
from datetime import datetime, timedelta

from startrek_client import Startrek
from dateutil.parser import parse as dt_parse
from dateutil.tz import tzutc

import infra.rtc.iolimit_ticketer.utils as utils

INACTIVITY_DAYS = 30


def get_ticket_tag(marker):
    if marker == "ssd":
        return "iolimit"
    elif marker == "hdd":
        return "iolimit_hdd"
    elif marker == "net":
        return "netlimit"
    elif marker == "qyp":
        return "qyp"
    else:
        raise Exception("unknown marker {}".format(marker))


def create_st_client():
    return Startrek(useragent="iolimit rtc ticketer", token=utils.get_st_oauth_token())


def has_activity(issue):
    last_activities = [dt_parse(issue.createdAt)]
    last_activities.extend(
        dt_parse(x.createdAt) for x in issue.comments if x.createdBy.id not in ["dldmitry", "glebskvortsov"]
    )
    last_activitity = max(last_activities)
    logging.info("Issue %r has last activity at %r", issue, last_activitity)
    return (datetime.now(tz=tzutc()) - last_activitity) <= timedelta(days=INACTIVITY_DAYS)


def get_ticket_map(client, marker, get_closed=False):
    result = {}

    if not get_closed:
        issues = client.issues.find(filter={'queue': 'RTCRESOURCES', 'tags': get_ticket_tag(marker),
                                            'resolution': 'empty()'}, per_page=100)
    else:
        issues = client.issues.find(filter={'queue': 'RTCRESOURCES', 'tags': get_ticket_tag(marker)}, per_page=100)

    for issue in issues:
        service_id = None
        for service in issue.abcService:
            service_id = service.id
        if service_id is not None:
            result[int(service_id)] = issue.key
    return result


def get_ticket_summary(marker, abc_name):
    if marker == "ssd":
        return u'Включить IO лимиты на чтение с SSD для сервисов {}'.format(abc_name.get("en") or abc_name.get("ru"))
    elif marker == "hdd":
        return u'Включить IO лимиты на чтение с HDD для сервисов {}'.format(abc_name.get("en") or abc_name.get("ru"))
    elif marker == "net":
        return u'Включить NET гарантии для сервисов {}'.format(abc_name.get("en") or abc_name.get("ru"))
    elif marker == "qyp":
        return u'Включить IO лимиты на ВМ QYP для сервисов {}'.format(abc_name.get("en") or abc_name.get("ru"))
    else:
        raise Exception("unknown marker {}".format(marker))


def get_or_create_ticket(client, ticket_map, abc_service_id, abc_services, marker, followers, dry_run=True):
    if abc_service_id in ticket_map:
        return ticket_map[abc_service_id]
    if dry_run:
        return None
    abc_name, abc_slug = abc_services[abc_service_id]
    tags = [get_ticket_tag(marker), 'abc_{}'.format(abc_slug)]

    issue = client.issues.create(
        queue='RTCRESOURCES',
        summary=get_ticket_summary(marker, abc_name),
        type={'name': 'Task'},
        abcService=abc_service_id,
        tags=tags,
        followers=sorted(followers)
    )
    time.sleep(1)
    return issue.key


def close_unrelevant_tickets(client, ticket_map, grouped_services, dry_run=True):
    for abc_service_id in ticket_map.viewkeys() - grouped_services.viewkeys():
        issue = client.issues[ticket_map[abc_service_id]]
        if dry_run:
            logging.warning("Ticket %r for abc service %d should be closed as unrelevant", issue, abc_service_id)
        else:
            transition = issue.transitions['close']
            transition.execute(resolution='fixed', comment=u'Тикет потерялся, закрываем его.')
