# coding: utf-8

from __future__ import absolute_import, print_function

import time
from collections import defaultdict
import pickle
import logging

from walle_api import WalleClient

WALLE_CACHE_VERSION = 2
WALLE_CACHE_TTL = 24 * 60 * 60


class WalleHost(object):

    __slots__ = ["fqdn", "project", "tags"]

    def __init__(self):
        self.fqdn = None
        self.project = None
        self.tags = None


def compute_stats(host_map):
    client = WalleClient()
    for host in client.iter_hosts(fields=["name", "project", "tags"], tags=["rtc"]):
        walle_host = host_map[host["name"]]
        walle_host.fqdn = host["name"]
        walle_host.project = host["project"]
        walle_host.tags = host["tags"]


class WalleStat(object):

    def __init__(self):
        self.host_map = defaultdict(WalleHost)

        self._version = WALLE_CACHE_VERSION
        self._timestamp = time.time()

    def freeze(self):
        self.host_map = dict(self.host_map)

    def check_actual(self):
        if self._version != WALLE_CACHE_VERSION:
            raise Exception("version mismatch")
        if time.time() - self._timestamp > WALLE_CACHE_TTL:
            raise Exception("cache expired")


def compute_walle_stat():
    stat = WalleStat()
    logging.info("Get walle stats")
    compute_stats(stat.host_map)
    stat.freeze()
    return stat


def get_walle_stat_cached():
    """
    :rtype: WalleStat
    """
    file_name = "walle_stat.tmp"
    try:
        with open(file_name, "rb") as stream:
            result = pickle.load(stream)  # type: WalleStat
            result.check_actual()
            return result
    except Exception:
        result = compute_walle_stat()
        result.check_actual()
        with open(file_name, "wb") as stream:
            pickle.dump(result, stream, protocol=pickle.HIGHEST_PROTOCOL)
        return result
