# -*- coding: utf-8 -*-

import os
import logging
import click
import json
import time

from infra.rtc.janitor.scenario import Scenario
from infra.rtc.janitor.context import Context
from infra.rtc.janitor.common import (
    add_command_name,
    get_tickets,
    get_parsed_ticket,
    get_hosts_info
    )
from infra.rtc.janitor.exceptions import InvalidParsedDataException
from infra.rtc.janitor.fsm_stages.common import (
    handle_scenario_stages,
    handle_new_scenario)
from infra.rtc.janitor.constants import (
    PROG_NAME,
    PROG_VER,
    ST_FILTER_STRING,
)
from infra.rtc.janitor.checks import (
    check_scenario_notstarted_timeout,
    send_iteration_status,
    select_scenarios_poweroff_queued,
    select_scenarios_overtimed)
from walle_api import WalleConnectionError

log = logging.getLogger(PROG_NAME)


class ListOptionInt(click.Option):
    def type_cast_value(self, ctx, value):
        if not value:
            return []
        try:
            return [int(i.strip()) for i in value.split(',')]
        except:  # noqa
            raise click.BadParameter(value)


class ListOptionStr(click.Option):
    def type_cast_value(self, ctx, value):
        if not value:
            return []
        try:
            return [i.strip() for i in value.split(',')]
        except:  # noqa
            raise click.BadParameter(value)


def print_dict(data):
    print(json.dumps(data, indent=4))


@click.group()
@click.option('--level', '-l', type=click.Choice(['error', 'info', 'debug']), default='error', help='Enables verbose mode.')
@click.option('--dry-run', 'dry_run', is_flag=True, help='Enable dry run mode.')
@click.option('--oauth', envvar='OAUTH', help='OAuth token.')
@click.option('--tvm', envvar='TVM', help='TVM token.')
@click.version_option(PROG_VER)
@click.pass_context
def main(ctx, level, oauth, tvm, dry_run):
    """
    """
    logging.basicConfig(level=level.upper(),
                        handlers=[logging.StreamHandler()],
                        format='%(asctime)s\t%(levelname)s\t%(name)-20s\t%(message)s')
    try:
        if not oauth:
            log.debug('Attempting get outh token from ~/.oauth')
            with open(os.path.expanduser('~/.oauth')) as f:
                oauth = f.readline().strip('\n')
    except IOError:
        raise RuntimeError("No OAuth token specified, use env OAUTH or put token to ~/.oauth")
    ctx.obj = Context(oauth, tvm, dry_run)


@main.command('add_hosts', short_help='Add hosts to wall-e project.')
@click.option('--hosts', required=True, cls=ListOptionInt, help='Hosts list, separated by comma.')
@click.option('--target_project_id', required=True, help='Wall-e project ID. ')
@click.option('--ticket_key', required=False, help='ST ticket.')
@click.option('--comment', default='', help='Comment.')
@click.option('--responsible', required=False, cls=ListOptionStr, help='Responsible list, separated by comma.')
@click.pass_context
@add_command_name
def cmd_add_hosts(ctx, **kwargs):
    """Add any hosts to project except hosts from preorders"""
    scenario = Scenario.action(ctx, **kwargs)
    log.info(scenario.to_dict())
    print(scenario.ticket_key)


@main.command('rm_hosts', short_help='Add or move exists hosts to wall-e project.')
@click.option('--hosts', required=True, cls=ListOptionInt, help='Hosts list, separated by comma.')
@click.option('--target_project_id', required=True, help='Wall-e project ID. ')
@click.option('--ticket_key', required=False, help='ST ticket.')
@click.option('--dismantle', is_flag=True, help='Dismantle hosts.')
@click.option('--comment', default='', help='Comment.')
@click.option('--responsible', default='', cls=ListOptionStr, help='Responsible list, separated by comma.')
@click.pass_context
@add_command_name
def cmd_rm_hosts(ctx, **kwargs):
    """Remove hosts from project"""
    scenario = Scenario.action(ctx, **kwargs)
    log.info(scenario.to_dict())
    print(scenario.ticket_key)


@main.command('list', short_help='List scenarios.')
@click.option('--id', 'scenario_id', default=None, help='ID scenario.')
@click.option('--status', default=None, help='ID scenario.')
@click.pass_context
def cmd_list(ctx, **kwargs):
    for scenario in Scenario.load_all(ctx.obj.walle_client, **kwargs):
        print_dict(scenario.to_dict())


@main.command('run', short_help='Run scenarios')
@click.option('--id', 'scenario_id', default=None, help='ID scenario.')
@click.option('--status', default="created,started,finished", help='Even if scenario is started.')
@click.option('--ticket_list', default=[], required=False, cls=ListOptionStr, help='Run against these ST ticket.')
@click.pass_context
def cmd_run(ctx, ticket_list, **kwargs):
    ctx.obj.counters['start_time'] = time.time()
    ctx.obj.scenarios_list = list(Scenario.load_all(
        ctx.obj.walle_client,
        fields=('scenario_id', 'hosts', 'issuer', 'labels', 'name', 'scenario_type', 'script_args', 'status', 'ticket_key', 'creation_time', 'action_time'),
        **kwargs))
    log.debug("%d active scenario found", len(ctx.obj.scenarios_list))

    tickets = get_tickets(ctx.obj, ctx.obj.scenarios_list, ST_FILTER_STRING)
    if ticket_list:
        tickets = [t for t in tickets if t.key in ticket_list]

    for ticket in tickets:
        try:
            parsed_ticket = get_parsed_ticket(ticket)
            ctx.obj.scenarios_list.append(Scenario.action(ctx, **parsed_ticket))
            ctx.obj.counters['parsed_tickets'] += 1
        except InvalidParsedDataException as ticket_err:
            log.warning('Skip ticket %s, %s', ticket.key, ticket_err)
            ctx.obj.counters['tickets_errors'].append('{}:{};'.format(ticket.key, ticket_err))
        except WalleConnectionError as e:
            log.exception(e)

    for scenario in ctx.obj.scenarios_list:
        if not scenario.labels.get("source") == PROG_NAME:
            log.warning("Scenario %s:%s doesn't belong me.", scenario.scenario_id, scenario.name)
            continue
        if scenario.fsm_processed:
            log.warning("Scenario %s:%s is already processed, skipping it", scenario.scenario_id, scenario.name)
            continue
        elif not scenario.fsm_stages:
            log.info("Scenario %s:%s is met for the first time", scenario.scenario_id, scenario.name)
            handle_new_scenario(ctx.obj.walle_client, scenario)
            ctx.obj.counters['processed_scenarios'] += 1
        log.debug("Start dispatching %r", scenario)
        try:
            handle_scenario_stages(ctx.obj, scenario)
        except Exception as scenario_err:
            log.exception("Scenario %r:%s processing finished with error: %s", scenario.scenario_id, scenario.name, scenario_err)
            ctx.obj.counters['scenarios_errors'].append('#{} {}[{}]:{};'.format(scenario.scenario_id, scenario.name, scenario.fsm_curr_stage, scenario_err))

    ctx.obj.counters['end_time'] = time.time()
    ctx.obj.counters['processing_time'] = ctx.obj.counters['end_time'] - ctx.obj.counters['start_time']
    ctx.obj.counters['scenarios_notstarted'] = check_scenario_notstarted_timeout(ctx.obj)
    ctx.obj.counters['scenarios_poweroff_queued'] = select_scenarios_poweroff_queued(ctx.obj)
    ctx.obj.counters['scenarios_overtime'] = list(select_scenarios_overtimed(ctx.obj))

    send_iteration_status(ctx.obj)


@main.command('finish', short_help='Finish scenarios.')
@click.option('--id', 'scenario_id', required=True, default=None, help='ID scenario.')
@click.pass_context
def cmd_finish(ctx, **kwargs):
    for scenario in Scenario.load_all(ctx.obj.walle_client, **kwargs):
        scenario.mark_as_processed()
        scenario.save(ctx.obj.walle_client)
        print_dict(scenario.to_dict())


# # TODO delete this (for debug only)
# @main.command('modify', short_help='Modify scenarios.')
# @click.option('--id', 'scenario_id', required=True, default=None, help='ID scenario.')
# @click.option('--labels', required=False, help='Modify labels of scenario.')
# @click.option('--script_args', required=False, help='Modify labels of scenario.')
# @click.option('--ticket_key', required=False, help='Modify ticket_key of scenario.')
# @click.pass_context
# def cmd_modify(ctx, scenario_id, script_args, labels, ticket_key):
#     for scenario in Scenario.load_all(ctx.obj.walle_client, scenario_id=scenario_id):
#         if labels:
#             scenario.labels = json.loads(labels)
#         if script_args:
#             scenario.script_args = json.loads(script_args)
#         if ticket_key:
#             scenario.ticket_key = ticket_key
#         scenario.save(ctx.obj.walle_client)
#         print_dict(scenario.to_dict())

# @main.command('preorder_add_hosts', short_help='Add hosts to wall-e project.')
# @click.option('--preorder_id', required=True, help='Hosts list, separated by comma.')
# @click.option('--hosts', required=False, cls=ListOptionInt, help='Hosts list, separated by comma.')
# @click.option('--target_project_id', required=True, help='Wall-e project ID. ')
# @click.option('--ticket_key', required=False, help='ST ticket.')
# @click.option('--comment', default='', help='Comment.')
# @click.option('--responsible', required=False, cls=ListOptionStr, help='Responsible list, separated by comma.')
# @click.pass_context
# @add_command_name
# #@validate_scenario
# @validate_ticket
# def preorder_add_hosts(ctx, **kwargs):
#     scenario = Scenario.action(ctx, **kwargs)
#     print_dict(scenario.to_dict())

# @main.command('move_hosts', short_help='Move exists hosts to wall-e project.')
# @click.option('--hosts', required=True, cls=ListOptionInt, help='Hosts list, separated by comma.')
# @click.option('--target_project_id', required=True, help='Wall-e project ID. ')
# @click.option('--ticket_key', required=False, help='ST ticket.')
# @click.option('--comment', default='', help='Comment.')
# @click.option('--responsible', required=False, cls=ListOptionStr, help='Responsible list, separated by comma.')
# @click.pass_context
# @add_command_name
# #@validate_scenario
# @validate_ticket
# def move_hosts(ctx, **kwargs):
#     scenario = Scenario.action(ctx, **kwargs)
#     print_dict(scenario.to_dict())

@main.command('resolve_hosts', short_help='Debug output for host`s resolving.')
@click.option('--hosts', required=True, cls=ListOptionInt, help='Hosts list, separated by comma.')
@click.pass_context
@add_command_name
def resolve_hosts(ctx, **kwargs):
    get_hosts_info(ctx.obj, kwargs['hosts'])
    print_dict(ctx.obj.hosts_info)


if __name__ == "__main__":
    main()
