# -*- coding: utf-8 -*-

import logging
from .constants import SCENARIO_NOTSTARTED_TIMEOUT_HOURS, SCENARIOS_POWEROFF_QUEUED_WARN, SCENARIOS_POWEROFF_QUEUED_CRIT, SCENARIO_EXECUTION_TIME_CRIT


log = logging.getLogger(__name__)

SCENARIO_NOTSTARTED_TIMEOUT = SCENARIO_NOTSTARTED_TIMEOUT_HOURS * 3600


def timeout(context, timestamp):
    return context.counters['start_time'] - timestamp > SCENARIO_NOTSTARTED_TIMEOUT


def check_scenario_notstarted_timeout(context):
    checked_tasks = {'add_hosts', 'add_hosts_qloud', 'move_hosts', 'rm_hosts'}
    return [s for s in context.scenarios_list if s.status == 'created' and timeout(context, s.creation_time) and s.labels.get('task_name', '') in checked_tasks]


def select_scenarios_poweroff_queued(context):
    checked_tasks = ('power_off',)
    return [s for s in context.scenarios_list if s.status == 'created' and s.labels.get('task_name', '') in checked_tasks]


def select_scenarios_overtimed(context):
    for scenario in context.scenarios_list:
        if scenario.status != 'started':
            continue
        if scenario.scenario_type not in SCENARIO_EXECUTION_TIME_CRIT:
            continue
        time_limit = SCENARIO_EXECUTION_TIME_CRIT[scenario.scenario_type]
        processing_time = scenario.action_time - context.counters['start_time']
        if time_limit <= processing_time:
            yield {
                'scenario': scenario,
                'time_limit': time_limit,
                'time_spent': processing_time
            }


def human_readable_time(timedelta):
    res = []
    days = timedelta // 86400
    if days:
        res.append('{}d'.format(days))
    hours = timedelta // 3600 % 24
    if hours:
        res.append('{}h'.format(hours))
    minutes = timedelta // 60 % 60
    if minutes:
        res.append('{}m'.format(minutes))
    # seconds = timedelta // 60
    # if seconds:
    #     res.append('{}s'.format(minutes))
    return ' '.join(res)


def send_iteration_status(context):
    if context.counters['scenarios_notstarted']:
        for scenario in context.counters['scenarios_notstarted']:
            log.debug("Scenario not started: %s", scenario)
        message = '{} scenarios not started too long: {}'.format(
            len(context.counters['scenarios_notstarted']),
            ', '.join(['#{} {} {}'.format(s.scenario_id, s.ticket_key, s.name) for s in context.counters['scenarios_notstarted']])
        )
        context.juggler_client.send_events_to_juggler('CRIT', message, 'scenarios_notstart')
    else:
        context.juggler_client.send_events_to_juggler('OK', '0 scenarios not started too long', 'scenarios_notstart')

    if len(context.counters['scenarios_poweroff_queued']) >= SCENARIOS_POWEROFF_QUEUED_WARN:
        for scenario in context.counters['scenarios_poweroff_queued']:
            log.debug("Scenario power_off queued: %s", scenario)
        message = '{} power_off scenarios queued: {}'.format(
            len(context.counters['scenarios_poweroff_queued']),
            ', '.join(['#{} {} {}'.format(s.scenario_id, s.ticket_key, s.name) for s in context.counters['scenarios_poweroff_queued']])
        )
        if len(context.counters['scenarios_poweroff_queued']) >= SCENARIOS_POWEROFF_QUEUED_CRIT:
            lvl = 'CRIT'
        else:
            lvl = 'WARN'
        context.juggler_client.send_events_to_juggler(lvl, message, 'scenarios_poweroff_queued')
    else:
        context.juggler_client.send_events_to_juggler('OK', '0 power_off scenarios queued', 'scenarios_poweroff_queued')

    if context.counters['tickets_errors']:
        for scenario in context.counters['tickets_errors']:
            log.debug("Ticket processed with errors: %s", scenario)
        message = '{} tickets processed with errors: {}'.format(
            len(context.counters['tickets_errors']),
            ', '.join(context.counters['tickets_errors'])
        )
        context.juggler_client.send_events_to_juggler('CRIT', message, 'tickets_errors')
    else:
        context.juggler_client.send_events_to_juggler('OK', '0 tickets processed with errors', 'tickets_errors')

    if context.counters['scenarios_errors']:
        for scenario in context.counters['scenarios_errors']:
            log.debug("Scenario processed with errors: %s", scenario)
        message = '{} scenarios processed with errors: {}'.format(
            len(context.counters['scenarios_errors']),
            ', '.join(context.counters['scenarios_errors'])
        )
        context.juggler_client.send_events_to_juggler('CRIT', message, 'scenarios_errors')
    else:
        context.juggler_client.send_events_to_juggler('OK', '0 scenarios processed with errors', 'scenarios_errors')

    if context.counters['scenarios_overtime']:
        # for item in context.counters['scenarios_overtime']:
        # log.debug("Scenario processed too long: %(scenario)s - %(time_spent)s (%(time_limit)s)", **item)
        message = '{} scenarios processed too long: {}'.format(
            len(context.counters['scenarios_overtime']),
            ', '.join(['#{} {} {}: {}({})'.format(
                item['scenario'].scenario_id,
                item['scenario'].ticket_key,
                item['scenario'].name,
                human_readable_time(item['time_spent']),
                human_readable_time(item['time_limit'])
            ) for item in context.counters['scenarios_overtime']])
        )
        context.juggler_client.send_events_to_juggler('CRIT', message, 'scenarios_overtime')
    else:
        context.juggler_client.send_events_to_juggler('OK', '0 scenarios processed too long', 'scenarios_overtime')
