# -*- coding: utf-8 -*-

import requests
import logging
from collections import defaultdict

log = logging.getLogger(__name__)


class AbcClient():
    __url = "https://abc-back.yandex-team.ru/api/v4"
    __headers = {"Content-Type": "application/json;charset=UTF-8"}

    def __init__(self, useragent, oauth_token):
        self.__cache_get_duty = {}
        self.__cache_get_members = defaultdict(lambda: defaultdict(list))
        self.__cache_services_info = {'slug': {}, 'id': {}, 'name': {}, 'name_en': {}}
        self.__headers["User-Agent"] = useragent
        self.__headers["Authorization"] = 'OAuth {}'.format(oauth_token)

    def call(self, method, hand, params=None, data=None):
        call_resp = []
        url = "{}/{}".format(self.__url, hand)
        log.debug('abc_call %r %r H:%r P:%r D:%r', method, url, self.__headers, params, data)
        resp = requests.request(method=method, url=url, headers=self.__headers, params=params, data=data)
        while True:
            call_resp.extend(resp.json()['results'])
            if resp.json()['next']:
                resp = requests.request(method=method, url=resp.json()['next'], headers=self.__headers, params=None, data=data)
            else:
                break
        return call_resp

    def call_singlepage(self, method, hand, params=None, data=None):
        url = "{}/{}".format(self.__url, hand)
        log.debug('abc_call %r %r H:%r P:%r D:%r', method, url, self.__headers, params, data)
        resp = requests.request(method=method, url=url, headers=self.__headers, params=params, data=data)
        return resp.json()

    def service_name_to_slug(self, service_name):
        return self.get_service_info(name=service_name)['slug']

    def service_slug_to_id(self, service_slug):
        return self.get_service_info(slug=service_slug)['id']

    def _get_filtered_param(self, kwargs):
        kwargs = {k: w for k, w in kwargs.iteritems() if k in ('slug', 'id', 'name', 'name_en')}
        if len(kwargs) != 1:
            raise ValueError('''only one parameter in 'slug', 'id', 'name', 'name_en' must be passed''')
        return kwargs.items()[0]

    def _get_service_info_from_abc(self, attr, value):
        params = {attr: value, 'fields': 'slug,id,name', 'state__in': 'develop,supported'}
        resp = self.call('GET', 'services/', params=params)[0]
        resp['name_en'] = resp['name']['en']
        resp['name'] = resp['name']['ru'].encode('utf-8')
        for k, v in resp.iteritems():
            self.__cache_services_info[k][v] = resp
        return resp

    def get_service_info(self, **kwargs):
        attr, value = self._get_filtered_param(kwargs)
        cached = self.__cache_services_info[attr].get(value)
        if not cached:
            cached = self._get_service_info_from_abc(attr, value)
        return cached

    def _get_members_remove_prefix(self, name):
        return name[8:]

    def _get_members_filtered_param(self, kwargs):
        kwargs = {self._get_members_remove_prefix(k): w for k, w in kwargs.iteritems() if k in ('service_id', 'service_slug', 'service_name', 'service_name_en')}
        if len(kwargs) != 1:
            raise ValueError('''only one parameter in 'service_id', 'service_slug', 'service_name', 'service_name_en' must be passed''')
        return kwargs

    def get_members(self, roles=[], **kwargs):  # service_id=None, service_slug=None, service_name=None,
        args = self._get_members_filtered_param(kwargs)
        cache_id = self.get_service_info(**args)['id']
        for role in roles:
            if role not in self.__cache_get_members[cache_id]:
                params = {
                    'fields': 'person',
                    'service__id': cache_id,
                    'role__scope': role
                }
                resp = self.call('GET', 'services/members/', params=params)
                if resp:
                    for i in resp:
                        self.__cache_get_members[cache_id][role].append(i["person"]["login"])
        return {role: self.__cache_get_members[cache_id][role] for role in roles}

    def get_duty(self, service_id):
        if service_id not in self.__cache_get_duty:
            url = 'services/{}/on_duty/'.format(service_id)
            resp = self.call_singlepage('GET', url)
            self.__cache_get_duty[service_id] = {
                'login': resp[0]['person']['login'],
                'name_en': resp[0]['person']['name']['en'],
                'name_ru': resp[0]['person']['name']['ru']
            }
        return self.__cache_get_duty[service_id]
