# -*- coding: utf-8 -*-

import requests
import logging
from requests.exceptions import RequestException

log = logging.getLogger(__name__)


class BotClientError(Exception):
    pass


class BotClientHostNotFound(BotClientError):
    pass


class BotClient():
    __url = "https://bot.yandex-team.ru/api"
    __headers = {"Content-Type": "application/json;charset=UTF-8"}

    def __init__(self, useragent, oauth_token):
        self.__headers["Authorization"] = 'OAuth {}'.format(oauth_token)
        self.__headers["User-Agent"] = useragent

    def call(self, method, hand, params=None, data=None):
        url = "{}/{}".format(self.__url, hand)
        try:
            resp = requests.request(method, url, headers=self.__headers, params=params, data=data)
        except RequestException as e:
            raise BotClientError(e)
        if not resp.ok:
            raise BotClientError(resp.json())
        return self._parse_resp(resp, params, data)

    def _parse_resp(self, resp, params, data):
        data = resp.json()
        if not data.get('msg'):
            return data
        if data['msg'] == 'not found':
            raise BotClientHostNotFound('Host with inv:{}, fqdn:{} not found'.format(params.get('inv'), params.get('fqdn')))
        else:
            raise BotClientError('Request url:{}, msg:{}'.format(resp.url, data['msg']))

    def get_host_info(self, inv=None, fqdn=None):
        host_info = {}
        params = {"output": "XXCSI_FQDN|GROUP_OWNED|loc_object|instance_number|loc_segment5"}
        if inv is not None and fqdn is None:
            params["inv"] = inv
        elif inv is None and fqdn is not None:
            params["fqdn"] = fqdn
        else:
            return host_info
        resp = self.call('GET', 'osinfo.php', params=params)
        host_info = {
            'inv':  resp["os"][0]["instance_number"],
            'fqdn': resp["os"][0]["XXCSI_FQDN"],
            'abc_prj': resp["os"][0]["GROUP_OWNED"],
            'dc': resp["os"][0]["loc_object"].lower(),
            'rack': resp["os"][0]["loc_segment5"],
        }
        return host_info

    def get_preorder_info(self, id):
        resp = self.call('GET', 'v1/hwr/pre-orders/{}'.format(id))
        current = resp["current_version"]
        order_info = {
            'status_id':  current["status_id"],
            'status_code': current["status_code"],
            'status_display': current["status_display"],
            'ticket_id': current["ticket_id"]
            }
        return order_info
